<?php
/**
 * Plugin Name: MetricPoints Error Tracker
 * Plugin URI: https://metricpoints.com
 * Description: Track JavaScript errors and performance metrics with MetricPoints
 * Version: 1.0.0
 * Author: Michael Loeffler
 * Author URI: https://mainstreetwebdeveloper.com
 * Requires at least: 5.8
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: metricpoints-error-tracker
 * Network: false
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('METRICPOINTS_ERROR_TRACKER_VERSION', '1.0.0');
define('METRICPOINTS_ERROR_TRACKER_PLUGIN_URL', plugin_dir_url(__FILE__));
define('METRICPOINTS_ERROR_TRACKER_PLUGIN_PATH', plugin_dir_path(__FILE__));

class MetricPointsErrorTracker {
    
    private $options;
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_footer', array($this, 'add_footer_script'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_footer', array($this, 'add_footer_script'));
        add_action('wp_ajax_metricpoints_error_verify_connection', array($this, 'handle_verify_connection'));
        
        // Load options
        $this->options = get_option('metricpoints_error_tracker_options', array());
    }
    
    public function init() {
        // Plugin initialization
    }
    
    public function add_admin_menu() {
        add_options_page(
            'MetricPoints Error Tracker',
            'MetricPoints Error Tracker',
            'manage_options',
            'metricpoints-error-tracker',
            array($this, 'admin_page')
        );
    }
    
    public function admin_init() {
        register_setting('metricpoints_error_tracker_options', 'metricpoints_error_tracker_options', array($this, 'validate_options'));
        
        // Settings are now manually rendered in admin_page()
    }
    
    /**
     * Validate and verify options on save
     */
    public function validate_options($input) {
        $validated = array();
        
        // Validate and verify API key if provided
        if (isset($input['api_key']) && !empty($input['api_key'])) {
            $api_key = sanitize_text_field($input['api_key']);
            
            // Verify API key with MetricPoints
            $verification = $this->verify_api_key($api_key);
            
            if ($verification['success']) {
                $validated['api_key'] = $api_key;
                // Store project info for display
                $validated['project_info'] = $verification['project'] ?? null;
                $validated['connection_status'] = 'connected';
                $validated['connection_verified_at'] = current_time('mysql');
                
                // Add success message
                add_settings_error('metricpoints_error_tracker_options', 'api_key_valid', '✅ API key verified successfully! Connected to project: ' . ($verification['project']['name'] ?? 'Unknown'), 'updated');
            } else {
                // Keep old API key if verification fails
                $old_options = get_option('metricpoints_error_tracker_options', array());
                $validated['api_key'] = $old_options['api_key'] ?? '';
                $validated['connection_status'] = 'failed';
                
                add_settings_error('metricpoints_error_tracker_options', 'api_key_invalid', '❌ API key verification failed: ' . ($verification['error'] ?? 'Invalid API key. Please check your API key and try again.'), 'error');
            }
        } elseif (isset($input['api_key']) && empty($input['api_key'])) {
            // API key cleared
            $validated['api_key'] = '';
            $validated['connection_status'] = 'disconnected';
            $validated['project_info'] = null;
        } else {
            // Keep existing API key
            $old_options = get_option('metricpoints_error_tracker_options', array());
            $validated['api_key'] = $old_options['api_key'] ?? '';
            $validated['project_info'] = $old_options['project_info'] ?? null;
            $validated['connection_status'] = $old_options['connection_status'] ?? 'unknown';
        }
        
        // Validate other options (pass through)
        foreach ($input as $key => $value) {
            if (!in_array($key, ['api_key', 'project_info', 'connection_status', 'connection_verified_at'])) {
                $validated[$key] = $value;
            }
        }
        
        return $validated;
    }
    
    /**
     * Verify API key with MetricPoints backend
     */
    private function verify_api_key($api_key) {
        if (empty($api_key)) {
            return array('success' => false, 'error' => 'API key is empty');
        }
        
        // Determine API base URL
        $is_local_dev = defined('WP_DEBUG') && WP_DEBUG && 
                        (strpos(home_url(), '127.0.0.1') !== false || 
                         strpos(home_url(), 'localhost') !== false ||
                         strpos(home_url(), '.local') !== false);
        
        if ($is_local_dev) {
            $api_base = 'http://host.docker.internal:8000/api';
        } else {
            $api_base = 'https://metricpoints.com/api';
        }
        
        $verify_url = $api_base . '/v1/verify-api-key/' . urlencode($api_key);
        
        $response = wp_remote_get($verify_url, array(
            'timeout' => 10,
            'headers' => array(
                'User-Agent' => 'MetricPoints-WordPress-Plugin/' . METRICPOINTS_ERROR_TRACKER_VERSION
            )
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => 'Failed to connect to MetricPoints: ' . $response->get_error_message()
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        
        if ($response_code === 200 && isset($response_data['success']) && $response_data['success']) {
            return array(
                'success' => true,
                'project' => $response_data['project'] ?? null,
                'team' => $response_data['team'] ?? null,
                'project_type' => $response_data['project_type'] ?? 'error_tracking',
            );
        } else {
            return array(
                'success' => false,
                'error' => $response_data['error'] ?? 'Invalid API key'
            );
        }
    }
    
    /**
     * AJAX handler for verifying connection
     */
    public function handle_verify_connection() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_error_tracker_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        $api_key = $this->options['api_key'] ?? '';
        
        if (empty($api_key)) {
            wp_send_json_error('API key not configured');
        }
        
        $verification = $this->verify_api_key($api_key);
        
        if ($verification['success']) {
            // Update stored project info
            $this->options['project_info'] = $verification['project'] ?? null;
            $this->options['connection_status'] = 'connected';
            $this->options['connection_verified_at'] = current_time('mysql');
            update_option('metricpoints_error_tracker_options', $this->options);
            
            wp_send_json_success(array(
                'message' => '✅ Connection verified successfully!',
                'project' => $verification['project'] ?? null,
            ));
        } else {
            wp_send_json_error(array(
                'message' => '❌ Connection failed: ' . ($verification['error'] ?? 'Unknown error'),
            ));
        }
    }
    
    
    public function api_key_callback() {
        $value = isset($this->options['api_key']) ? $this->options['api_key'] : '';
        echo '<input type="text" name="metricpoints_error_tracker_options[api_key]" id="metricpoints-api-key" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">Enter your MetricPoints API key. Get one at <a href="https://metricpoints.com" target="_blank">metricpoints.com</a></p>';
        echo '<p class="description" style="color: #666;">💡 Tip: You can paste the full URL from your MetricPoints dashboard - we\'ll automatically extract the API key for you!</p>';
    }
    
    
    public function enabled_callback() {
        $value = isset($this->options['enabled']) ? $this->options['enabled'] : 0;
        echo '<label><input type="checkbox" name="metricpoints_error_tracker_options[enabled]" value="1" ' . checked(1, $value, false) . ' /> Enable error tracking on your website</label>';
    }
    
    public function admin_tracking_callback() {
        $value = isset($this->options['admin_tracking']) ? $this->options['admin_tracking'] : 0;
        echo '<label><input type="checkbox" name="metricpoints_error_tracker_options[admin_tracking]" value="1" ' . checked(1, $value, false) . ' /> Enable error tracking in WordPress admin area</label>';
        echo '<p class="description">Track JavaScript errors and performance metrics in the WordPress admin interface</p>';
    }
    
    public function auto_start_callback() {
        $value = isset($this->options['auto_start']) ? $this->options['auto_start'] : 1;
        echo '<label><input type="checkbox" name="metricpoints_error_tracker_options[auto_start]" value="1" ' . checked(1, $value, false) . ' /> Automatically start tracking when the script loads</label>';
    }
    
    public function auto_start_delay_callback() {
        $value = isset($this->options['auto_start_delay']) ? $this->options['auto_start_delay'] : 1000;
        echo '<input type="number" name="metricpoints_error_tracker_options[auto_start_delay]" value="' . esc_attr($value) . '" class="small-text" min="0" max="10000" />';
        echo '<p class="description">Delay in milliseconds before auto-starting (default: 1000ms)</p>';
    }
    
    public function performance_tracking_callback() {
        $value = isset($this->options['performance_tracking']) ? $this->options['performance_tracking'] : 1;
        echo '<label><input type="checkbox" name="metricpoints_error_tracker_options[performance_tracking]" value="1" ' . checked(1, $value, false) . ' /> Track performance metrics (Core Web Vitals, etc.)</label>';
    }
    
    public function user_tracking_callback() {
        $value = isset($this->options['user_tracking']) ? $this->options['user_tracking'] : 1;
        echo '<label><input type="checkbox" name="metricpoints_error_tracker_options[user_tracking]" value="1" ' . checked(1, $value, false) . ' /> Track user sessions and behavior</label>';
    }
    
    public function session_replay_callback() {
        $value = isset($this->options['session_replay']) ? $this->options['session_replay'] : 1;
        echo '<label><input type="checkbox" name="metricpoints_error_tracker_options[session_replay]" value="1" ' . checked(1, $value, false) . ' /> Enable session replay (requires user tracking)</label>';
    }
    
    public function network_monitoring_callback() {
        $value = isset($this->options['network_monitoring']) ? $this->options['network_monitoring'] : 1;
        echo '<label><input type="checkbox" name="metricpoints_error_tracker_options[network_monitoring]" value="1" ' . checked(1, $value, false) . ' /> Monitor network requests and failures</label>';
    }
    
    public function environment_callback() {
        $value = isset($this->options['environment']) ? $this->options['environment'] : 'production';
        echo '<select name="metricpoints_error_tracker_options[environment]" class="regular-text">';
        echo '<option value="production"' . selected('production', $value, false) . '>Production</option>';
        echo '<option value="staging"' . selected('staging', $value, false) . '>Staging</option>';
        echo '<option value="development"' . selected('development', $value, false) . '>Development</option>';
        echo '<option value="testing"' . selected('testing', $value, false) . '>Testing</option>';
        echo '</select>';
        echo '<p class="description">Set the environment for your WordPress site to help organize data in MetricPoints</p>';
    }
    
    public function diagnostic_logging_callback() {
        $value = isset($this->options['diagnostic_logging']) ? $this->options['diagnostic_logging'] : 0;
        echo '<label><input type="checkbox" name="metricpoints_error_tracker_options[diagnostic_logging]" value="1" ' . checked(1, $value, false) . ' /> Enable detailed diagnostic logging (development only)</label>';
        echo '<p class="description">Only enable for development and testing - not recommended for production</p>';
    }
    
    public function silent_mode_callback() {
        $value = isset($this->options['silent_mode']) ? $this->options['silent_mode'] : 0;
        echo '<label><input type="checkbox" name="metricpoints_error_tracker_options[silent_mode]" value="1" ' . checked(1, $value, false) . ' /> Enable silent mode (totally quiet)</label>';
        echo '<p class="description">Disable all console logging - only use for development and testing</p>';
    }
    
    
    public function admin_page() {
        ?>
        <div class="wrap">
            <h1>MetricPoints Error Tracker</h1>
            
            <!-- Connection Status Card -->
            <div class="card" style="margin-bottom: 20px;">
                <h2>🔗 Connection Status</h2>
                <?php 
                $connection_status = $this->options['connection_status'] ?? 'unknown';
                $api_key = $this->options['api_key'] ?? '';
                $project_info = $this->options['project_info'] ?? null;
                
                if (!empty($api_key)):
                    if ($connection_status === 'connected' && $project_info):
                        ?>
                        <div style="background: #d4edda; padding: 15px; border-radius: 5px; border-left: 4px solid #28a745; margin: 10px 0;">
                            <p style="margin: 0; color: #155724;"><strong>✅ Connected to MetricPoints</strong></p>
                            <p style="margin: 5px 0 0 0; color: #155724;">
                                <strong>Project:</strong> <?php echo esc_html($project_info['name'] ?? 'Unknown'); ?><br>
                                <strong>Domain:</strong> <?php echo esc_html($project_info['domain'] ?? 'Unknown'); ?><br>
                                <?php if (isset($this->options['connection_verified_at'])): ?>
                                    <strong>Last Verified:</strong> <?php echo esc_html(date('M j, Y g:i A', strtotime($this->options['connection_verified_at']))); ?>
                                <?php endif; ?>
                            </p>
                        </div>
                        <?php
                    else:
                        ?>
                        <div style="background: #fff3cd; padding: 15px; border-radius: 5px; border-left: 4px solid #ffc107; margin: 10px 0;">
                            <p style="margin: 0; color: #856404;"><strong>⚠️ Connection Status Unknown</strong></p>
                            <p style="margin: 5px 0 0 0; color: #856404;">API key is set but connection status is not verified. Click "Verify Connection" to check.</p>
                            <button id="verify-connection-btn" class="button" style="margin-top: 10px;">🔄 Verify Connection</button>
                            <div id="verify-connection-result" style="margin-top: 10px;"></div>
                        </div>
                        <?php
                    endif;
                else:
                    ?>
                    <div style="background: #f8d7da; padding: 15px; border-radius: 5px; border-left: 4px solid #dc3545; margin: 10px 0;">
                        <p style="margin: 0; color: #721c24;"><strong>❌ Not Connected</strong></p>
                        <p style="margin: 5px 0 0 0; color: #721c24;">Please enter your MetricPoints API key below to connect your account.</p>
                    </div>
                    <?php
                endif;
                ?>
            </div>
            
            <div class="card">
                <h2>Status</h2>
                <?php if (!empty($this->options['api_key']) && !empty($this->options['enabled'])): ?>
                    <p style="color: green;">✅ Error tracking is active</p>
                    <p><strong>API Key:</strong> <?php echo esc_html(substr($this->options['api_key'], 0, 8) . '...'); ?></p>
                    <p><strong>Environment:</strong> <?php echo esc_html($this->options['environment'] ?? 'production'); ?></p>
                    <p><strong>Frontend Tracking:</strong> <?php echo !empty($this->options['enabled']) ? 'Enabled' : 'Disabled'; ?></p>
                    <p><strong>Admin Tracking:</strong> <?php echo !empty($this->options['admin_tracking']) ? 'Enabled' : 'Disabled'; ?></p>
                    <p><strong>Diagnostic Logging:</strong> <?php echo !empty($this->options['diagnostic_logging']) ? 'Enabled' : 'Disabled'; ?></p>
                    <p><strong>Silent Mode:</strong> <?php echo !empty($this->options['silent_mode']) ? 'Enabled' : 'Disabled'; ?></p>
                <?php else: ?>
                    <p style="color: red;">❌ Error tracking is not configured or disabled</p>
                    <?php if (empty($this->options['api_key'])): ?>
                        <p style="color: orange;">⚠️ API Key is missing</p>
                    <?php endif; ?>
                    <?php if (empty($this->options['enabled'])): ?>
                        <p style="color: orange;">⚠️ Error tracking is disabled</p>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('metricpoints_error_tracker_options');
                ?>
                
                <!-- Basic Settings -->
                <table class="form-table">
                    <tbody>
                        <tr>
                            <th scope="row">API Key</th>
                            <td><?php $this->api_key_callback(); ?></td>
                        </tr>
                        <tr>
                            <th scope="row">Enable Error Tracking</th>
                            <td><?php $this->enabled_callback(); ?></td>
                        </tr>
                        <tr>
                            <th scope="row">Enable Admin Tracking</th>
                            <td><?php $this->admin_tracking_callback(); ?></td>
                        </tr>
                    </tbody>
                </table>
                
                <!-- Advanced Settings Toggle -->
                <div style="margin: 20px 0;">
                    <h2>Advanced Configuration <span id="advanced-toggle" style="cursor: pointer; color: #0073aa;">▼</span></h2>
                </div>
                
                <!-- Advanced Settings (Hidden by default) -->
                <div id="advanced-settings" style="display: none;">
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row">Auto Start</th>
                                <td><?php $this->auto_start_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Auto Start Delay</th>
                                <td><?php $this->auto_start_delay_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Enable Performance Tracking</th>
                                <td><?php $this->performance_tracking_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Enable User Tracking</th>
                                <td><?php $this->user_tracking_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Enable Session Replay</th>
                                <td><?php $this->session_replay_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Enable Network Monitoring</th>
                                <td><?php $this->network_monitoring_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Environment</th>
                                <td><?php $this->environment_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Enable Diagnostic Logging</th>
                                <td><?php $this->diagnostic_logging_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Enable Silent Mode</th>
                                <td><?php $this->silent_mode_callback(); ?></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                
                <?php submit_button(); ?>
            </form>
            
            <div class="card">
                <h2>Test Error Tracking</h2>
                <p>Click the button below to generate a test error and verify your configuration:</p>
                <?php if (!empty($this->options['api_key'])): ?>
                    <button id="test-error-btn" class="button button-primary">Generate Test Error</button>
                <?php else: ?>
                    <button id="test-error-btn" class="button button-primary" disabled>Generate Test Error</button>
                    <p class="description" style="color: #d63638;">Please configure your API Key first before testing.</p>
                <?php endif; ?>
                <div id="test-result" style="margin-top: 10px;"></div>
            </div>
            
            <div class="card">
                <h2>Documentation</h2>
                <p>For more information about MetricPoints Error Tracker, visit our <a href="https://metricpoints.com/docs/error-tracking" target="_blank">documentation</a>.</p>
            </div>
        </div>
        
        <script>
        // Verify Connection Button
        const verifyConnectionBtn = document.getElementById('verify-connection-btn');
        if (verifyConnectionBtn) {
            verifyConnectionBtn.addEventListener('click', function() {
                const resultDiv = document.getElementById('verify-connection-result');
                const btn = this;
                const originalText = btn.textContent;
                
                btn.disabled = true;
                btn.textContent = '⏳ Verifying...';
                
                if (resultDiv) {
                    resultDiv.innerHTML = '<div style="padding: 10px; background: #fff3cd; border-radius: 4px;">Verifying connection...</div>';
                }
                
                fetch(ajaxurl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=metricpoints_error_verify_connection&nonce=<?php echo wp_create_nonce('metricpoints_error_tracker_nonce'); ?>'
                })
                .then(response => response.json())
                .then(data => {
                    btn.disabled = false;
                    btn.textContent = originalText;
                    
                    if (resultDiv) {
                        if (data.success) {
                            resultDiv.innerHTML = '<div style="padding: 10px; background: #d4edda; border-radius: 4px; color: #155724;"><strong>✅ ' + data.data.message + '</strong>';
                            if (data.data.project) {
                                resultDiv.innerHTML += '<br>Project: ' + (data.data.project.name || 'Unknown');
                            }
                            resultDiv.innerHTML += '</div>';
                            
                            // Reload page after 2 seconds to show updated status
                            setTimeout(function() {
                                window.location.reload();
                            }, 2000);
                        } else {
                            resultDiv.innerHTML = '<div style="padding: 10px; background: #f8d7da; border-radius: 4px; color: #721c24;"><strong>❌ ' + (data.data?.message || 'Verification failed') + '</strong></div>';
                        }
                    }
                })
                .catch(error => {
                    btn.disabled = false;
                    btn.textContent = originalText;
                    
                    if (resultDiv) {
                        resultDiv.innerHTML = '<div style="padding: 10px; background: #f8d7da; border-radius: 4px; color: #721c24;"><strong>❌ Error: ' + error.message + '</strong></div>';
                    }
                });
            });
        }
        
        // Auto-extract API key from pasted URL
        function extractApiKey(value) {
            // Handle various URL patterns:
            // - https://metricpoints.com/api/error-reports/err_xxxxx
            // - @https://metricpoints.com/api/error-reports/err_xxxxx
            // - metricpoints.com/api/error-reports/err_xxxxx
            // - /api/error-reports/err_xxxxx
            
            const patterns = [
                /@?https?:\/\/[^\/]*metricpoints\.com\/api\/error-reports\/(err_[a-zA-Z0-9_.]+)/,
                /@?[^\/]*metricpoints\.com\/api\/error-reports\/(err_[a-zA-Z0-9_.]+)/,
                /\/api\/error-reports\/(err_[a-zA-Z0-9_.]+)/
            ];
            
            for (let pattern of patterns) {
                const match = value.match(pattern);
                if (match && match[1]) {
                    return match[1];
                }
            }
            
            return null;
        }
        
        function showExtractionConfirmation(input) {
            const originalColor = input.style.backgroundColor;
            input.style.backgroundColor = '#d4edda';
            setTimeout(function() {
                input.style.backgroundColor = originalColor;
            }, 1000);
        }
        
        document.getElementById('metricpoints-api-key').addEventListener('paste', function(e) {
            setTimeout(function() {
                const input = document.getElementById('metricpoints-api-key');
                let value = input.value;
                
                // Check if the pasted value looks like a URL
                if (value.includes('metricpoints.com') && value.includes('/api/error-reports/')) {
                    const apiKey = extractApiKey(value);
                    if (apiKey) {
                        input.value = apiKey;
                        showExtractionConfirmation(input);
                    }
                }
            }, 10); // Small delay to allow paste to complete
        });
        
        // Also handle manual typing/input changes
        document.getElementById('metricpoints-api-key').addEventListener('input', function(e) {
            const value = e.target.value;
            
            // Check if the value looks like a URL and extract API key
            if (value.includes('metricpoints.com') && value.includes('/api/error-reports/')) {
                const apiKey = extractApiKey(value);
                if (apiKey) {
                    e.target.value = apiKey;
                    showExtractionConfirmation(e.target);
                }
            }
        });
        
        document.getElementById('test-error-btn').addEventListener('click', function() {
            // Check if button is disabled
            if (this.disabled) {
                return;
            }
            
            const resultDiv = document.getElementById('test-result');
            resultDiv.innerHTML = '<p>Generating test error...</p>';
            
            // Check if MetricPoints Error Tracker is loaded
            if (typeof window.MetricPointsErrorTracker === 'undefined') {
                resultDiv.innerHTML = '<p style="color: red;">❌ MetricPoints Error Tracker is not loaded. Make sure the plugin is enabled and your API key is configured.</p><p style="color: gray; font-size: 12px;">Check the browser console for detailed loading information.</p>';
                return;
            }
            
            // Generate a test error that will be caught by the error tracker
            setTimeout(function() {
                resultDiv.innerHTML = '<p style="color: green;">✅ Test error generated! Check your MetricPoints dashboard to see if it was captured.</p>';
                
                // Create an unhandled error that the error tracker will catch
                // Don't wrap in try/catch - let it be truly unhandled
                setTimeout(function() {
                    throw new Error('Test error from MetricPoints WordPress plugin - ' + new Date().toISOString());
                }, 50);
                
                // Also try to manually trigger an error report if available
                if (window.MetricPointsErrorTracker && window.MetricPointsErrorTracker.reportError) {
                    try {
                        window.MetricPointsErrorTracker.reportError(new Error('Manual test error from WordPress plugin'), {
                            source: 'wordpress-plugin-test',
                            testMode: true
                        });
                    } catch (e) {
                        // Ignore if reportError doesn't exist or fails
                    }
                }
            }, 100);
        });
        
        // Advanced settings toggle
        document.getElementById('advanced-toggle').addEventListener('click', function() {
            const toggle = document.getElementById('advanced-toggle');
            const settings = document.getElementById('advanced-settings');
            
            if (settings.style.display === 'none') {
                settings.style.display = 'block';
                toggle.textContent = '▲';
            } else {
                settings.style.display = 'none';
                toggle.textContent = '▼';
            }
        });
        </script>
        <?php
    }
    
    public function enqueue_scripts() {
        $is_admin = is_admin();
        
        if (empty($this->options['enabled']) || empty($this->options['api_key'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints Plugin: Not loading - missing enabled or api_key');
            }
            return;
        }
        
        // Check if we should load in admin
        if ($is_admin && empty($this->options['admin_tracking'])) {
            return;
        }
        
        // Add the initialization script that will load MetricPoints
        wp_add_inline_script('jquery', $this->get_initialization_script(), 'after');
    }
    
    public function add_footer_script() {
        $is_admin = is_admin();

        if (empty($this->options['enabled']) || empty($this->options['api_key'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints Plugin: Footer script not added - missing enabled or api_key');
            }
            return;
        }
        
        // Check if we should load in admin
        if ($is_admin && empty($this->options['admin_tracking'])) {
            return;
        }
        
        echo '<script type="text/javascript">';
        echo $this->get_initialization_script();
        echo '</script>';
    }
    
    private function get_initialization_script() {
        $config = array(
            'apiKey' => $this->options['api_key'],
            'autoStart' => !empty($this->options['auto_start']),
            'autoStartDelay' => isset($this->options['auto_start_delay']) ? (int)$this->options['auto_start_delay'] : 1000,
            'environment' => isset($this->options['environment']) ? $this->options['environment'] : 'production',
            'diagnosticLogging' => !empty($this->options['diagnostic_logging']),
            'silent' => !empty($this->options['silent_mode']),
            'errorLogging' => !empty($this->options['silent_mode']) ? false : true,
            'userTracking' => array(
                'enabled' => !empty($this->options['user_tracking']),
                'fingerprinting' => true,
                'localStorage' => true,
                'privacyMode' => false,
            ),
            'performance' => array(
                'enabled' => !empty($this->options['performance_tracking']),
                'sendToPerformanceAPI' => true,
                'memoryCheckInterval' => 30000,
            ),
            'sessionReplay' => array(
                'enabled' => !empty($this->options['session_replay']) && !empty($this->options['user_tracking']),
                'mouseTracking' => true,
                'clickTracking' => true,
                'maxSendingEvents' => 1000,
                'sendToPerformanceAPI' => true,
            ),
            'network' => array(
                'enabled' => !empty($this->options['network_monitoring']),
            )
        );
        
        $diagnosticLogging = !empty($this->options['diagnostic_logging']);
        
        return "
        " . ($diagnosticLogging ? "console.log('MetricPoints WordPress Plugin: Script starting...');" : "") . "
        
        (function() {
            " . ($diagnosticLogging ? "console.log('MetricPoints WordPress Plugin: Starting initialization...');" : "") . "
            
            var script = document.createElement('script');
            script.src = 'https://metricpoints.com/js/metric-points-error-tracker.min.js';
            script.async = true;
            
            script.onload = function() {
                " . ($diagnosticLogging ? "console.log('MetricPoints WordPress Plugin: Script loaded, checking for objects...');" : "") . "
                
                if (window.MetricPointsErrorTracker && window.MetricPointsErrorTrackerConfig) {
                    " . ($diagnosticLogging ? "console.log('MetricPoints WordPress Plugin: Objects found, initializing...');" : "") . "
                    " . ($diagnosticLogging ? "console.log('MetricPoints WordPress Plugin: Config:', " . json_encode($config) . ");" : "") . "
                    
                    try {
                        window.MetricPointsErrorTracker.init(new window.MetricPointsErrorTrackerConfig(" . json_encode($config) . "));
                        " . ($diagnosticLogging ? "console.log('MetricPoints WordPress Plugin: Successfully initialized!');" : "") . "
                    } catch (error) {
                        console.error('MetricPoints WordPress Plugin: Initialization failed:', error);
                    }
                } else {
                    console.error('MetricPoints WordPress Plugin: MetricPointsErrorTracker or MetricPointsErrorTrackerConfig not found');
                    " . ($diagnosticLogging ? "console.log('MetricPoints WordPress Plugin: Available objects:', Object.keys(window).filter(key => key.toLowerCase().includes('metric')));" : "") . "
                }
            };
            
            script.onerror = function() {
                console.error('MetricPoints WordPress Plugin: Failed to load script from https://metricpoints.com/js/metric-points-error-tracker.min.js');
            };
            
            document.head.appendChild(script);
            " . ($diagnosticLogging ? "console.log('MetricPoints WordPress Plugin: Script element added to head');" : "") . "
        })();
        
        " . ($diagnosticLogging ? "console.log('MetricPoints WordPress Plugin: Script end');" : "") . "
        ";
    }
    
    // Removed handle_error_report() - now using direct MetricPoints API
}

// Initialize the plugin
new MetricPointsErrorTracker();
