<?php
/**
 * Plugin Name: MetricPoints CSP Manager
 * Plugin URI: https://metricpoints.com
 * Description: Self-healing Content Security Policy management with MetricPoints
 * Version: 1.0.0
 * Author: Michael Loeffler
 * Author URI: https://mainstreetwebdeveloper.com
 * Requires at least: 5.8
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: metricpoints-csp-manager
 * Network: false
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('METRICPOINTS_CSP_MANAGER_VERSION', '1.0.0');
define('METRICPOINTS_CSP_MANAGER_PLUGIN_URL', plugin_dir_url(__FILE__));
define('METRICPOINTS_CSP_MANAGER_PLUGIN_PATH', plugin_dir_path(__FILE__));

class MetricPointsCSPManager {
    
    private $options;
    private $csp_violations = array();
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
        add_action('init', array($this, 'add_csp_header'));
        add_action('wp_ajax_metricpoints_csp_violation', array($this, 'handle_csp_violation'));
        add_action('wp_ajax_nopriv_metricpoints_csp_violation', array($this, 'handle_csp_violation'));
        add_action('wp_ajax_metricpoints_rollback_policy', array($this, 'handle_rollback_policy'));
        add_action('wp_ajax_metricpoints_force_update', array($this, 'handle_force_update'));
        add_action('wp_ajax_metricpoints_reset_learning', array($this, 'handle_reset_learning'));
        add_action('wp_ajax_metricpoints_get_backups', array($this, 'handle_get_backups'));
        add_action('wp_ajax_metricpoints_analyze_violations', array($this, 'handle_analyze_violations'));
        add_action('wp_ajax_metricpoints_activate_skipped_update', array($this, 'handle_activate_skipped_update'));
        add_action('wp_ajax_metricpoints_test_webhook', array($this, 'handle_test_webhook'));
        add_action('wp_ajax_metricpoints_verify_connection', array($this, 'handle_verify_connection'));
        add_action('wp_ajax_metricpoints_register', array($this, 'handle_register'));
        add_action('wp_ajax_metricpoints_save_headers', array($this, 'handle_save_headers'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('rest_api_init', array($this, 'register_webhook_endpoint'));
        
        // Schedule periodic header saving
        add_action('metricpoints_save_header_history', array($this, 'save_header_history_periodic'));
        if (!wp_next_scheduled('metricpoints_save_header_history')) {
            wp_schedule_event(time(), 'daily', 'metricpoints_save_header_history');
        }
        
        // Check if setup is needed
        add_action('admin_notices', array($this, 'show_setup_notice'));
        add_action('wp_ajax_metricpoints_skip_setup', array($this, 'handle_skip_setup'));
        
        // Load options
        $this->options = get_option('metricpoints_csp_manager_options', array());
        
        // Load existing CSP violations
        $this->csp_violations = get_option('metricpoints_csp_violations', array());
    }
    
    public function init() {
        // Plugin initialization
    }
    
    /**
     * Show setup notice if plugin is not configured
     */
    public function show_setup_notice() {
        // Only show on admin pages and if user has permission
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Check if plugin is configured
        $api_key = isset($this->options['api_key']) ? $this->options['api_key'] : '';
        $enabled = isset($this->options['enabled']) ? $this->options['enabled'] : 0;
        
        // Check if setup was skipped
        if (get_transient('metricpoints_setup_skipped')) {
            return;
        }
        
        // Show setup notice if API key is missing or plugin is disabled
        if (empty($api_key) || !$enabled) {
            $setup_url = admin_url('options-general.php?page=metricpoints-test');
            ?>
            <div class="notice notice-info is-dismissible" id="metricpoints-setup-notice">
                <div style="display: flex; align-items: center; gap: 15px;">
                    <div style="flex-shrink: 0;">
                        <span style="font-size: 24px;">🛡️</span>
                    </div>
                    <div style="flex: 1;">
                        <h3 style="margin: 0 0 5px 0;">MetricPoints CSP Manager Ready to Configure</h3>
                        <p style="margin: 0 0 10px 0;">
                            Your self-healing Content Security Policy manager is installed and ready to protect your website. 
                            Complete the setup to enable automatic CSP violation monitoring and policy updates.
                        </p>
                        <div style="display: flex; gap: 10px; align-items: center;">
                            <a href="<?php echo esc_url($setup_url); ?>" class="button button-primary">
                                🚀 Complete Setup
                            </a>
                            <button type="button" class="button" onclick="metricpointsSkipSetup()">
                                Skip for Now
                            </button>
                            <a href="https://metricpoints.com" target="_blank" class="button button-secondary">
                                Learn More
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            
            <script>
            function metricpointsSkipSetup() {
                // Hide the notice
                document.getElementById('metricpoints-setup-notice').style.display = 'none';
                
                // Send AJAX request to skip setup
                fetch(ajaxurl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=metricpoints_skip_setup&nonce=<?php echo wp_create_nonce('metricpoints_skip_setup'); ?>'
                });
            }
            </script>
            <?php
        }
    }
    
    /**
     * Handle skip setup AJAX request
     */
    public function handle_skip_setup() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_skip_setup')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        // Mark setup as skipped (don't show notice again for 7 days)
        set_transient('metricpoints_setup_skipped', true, 7 * DAY_IN_SECONDS);
        
        wp_die('Setup skipped');
    }
    
    public function add_admin_menu() {
        add_options_page(
            'MetricPoints CSP Manager',
            'MetricPoints CSP Manager',
            'manage_options',
            'metricpoints-csp-manager',
            array($this, 'admin_page')
        );
    }
    
    public function admin_init() {
        register_setting('metricpoints_csp_manager_options', 'metricpoints_csp_manager_options', array($this, 'validate_options'));
        
        add_settings_section(
            'metricpoints_csp_manager_main',
            'Configuration',
            array($this, 'settings_section_callback'),
            'metricpoints-csp-manager'
        );
        
        add_settings_field(
            'api_key',
            'API Key',
            array($this, 'api_key_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        
        add_settings_field(
            'enabled',
            'Enable CSP Management',
            array($this, 'enabled_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'self_healing',
            'Enable Self-Healing',
            array($this, 'self_healing_callback'),
            'metricpoints-csp-manager',
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
                
        add_settings_field(
            'csp_policy',
            'CSP Policy',
            array($this, 'csp_policy_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'csp_mode',
            'CSP Mode',
            array($this, 'csp_mode_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        
        add_settings_field(
            'hsts_enabled',
            'Strict-Transport-Security (HSTS)',
            array($this, 'hsts_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'x_frame_options',
            'X-Frame-Options',
            array($this, 'x_frame_options_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'x_content_type_options',
            'X-Content-Type-Options',
            array($this, 'x_content_type_options_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'referrer_policy',
            'Referrer-Policy',
            array($this, 'referrer_policy_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'x_xss_protection',
            'X-XSS-Protection',
            array($this, 'x_xss_protection_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'permissions_policy',
            'Permissions-Policy',
            array($this, 'permissions_policy_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'expect_ct',
            'Expect-CT',
            array($this, 'expect_ct_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'clear_site_data',
            'Clear-Site-Data',
            array($this, 'clear_site_data_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'cross_origin_headers',
            'Cross-Origin Headers',
            array($this, 'cross_origin_headers_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'custom_headers',
            'Custom Headers',
            array($this, 'custom_headers_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'admin_headers',
            'Enable Headers in Admin',
            array($this, 'admin_headers_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'backup_limit',
            'Policy Backup Limit',
            array($this, 'backup_limit_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
        
        add_settings_field(
            'confidence_threshold',
            'Policy Update Confidence Threshold',
            array($this, 'confidence_threshold_callback'),
            'metricpoints-csp-manager',
            'metricpoints_csp_manager_main'
        );
    }
    
    public function settings_section_callback() {
        echo '<p>Configure your MetricPoints CSP Manager settings below.</p>';
    }
    
    public function api_key_callback() {
        $value = isset($this->options['api_key']) ? $this->options['api_key'] : '';
        echo '<input type="text" name="metricpoints_csp_manager_options[api_key]" id="metricpoints-csp-api-key" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">Enter your MetricPoints API key. Get one at <a href="https://metricpoints.com" target="_blank">metricpoints.com</a></p>';
        echo '<p class="description" style="color: #666;">💡 Tip: You can paste the full URL from your MetricPoints dashboard - we\'ll automatically extract the API key for you!</p>';
    }
    
    
    public function enabled_callback() {
        $value = isset($this->options['enabled']) ? $this->options['enabled'] : 0;
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[enabled]" value="1" ' . checked(1, $value, false) . ' /> Enable CSP management on your website</label>';
    }
    
    public function self_healing_callback() {
        $value = isset($this->options['self_healing']) ? $this->options['self_healing'] : 0;
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[self_healing]" value="1" ' . checked(1, $value, false) . ' /> Enable AI-Powered Self-Healing CSP</label>';
        echo '<p class="description">🚀 <strong>Innovative Feature:</strong> Automatically analyzes violations, learns patterns, and intelligently updates your CSP policy in real-time. No more manual policy maintenance!</p>';
    }
    
    
    public function csp_policy_callback() {
        $value = isset($this->options['csp_policy']) ? $this->options['csp_policy'] : $this->get_default_csp_policy();
        
        // Validate the current policy
        $validation = $this->validate_csp_policy($value);
        
        echo '<textarea name="metricpoints_csp_manager_options[csp_policy]" id="csp-policy-textarea" rows="10" cols="50" class="large-text">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">Your Content Security Policy. The report-uri directive will be automatically added as <code>https://metricpoints.com/api/csp-reports/{your-api-key}</code> when the policy is applied.</p>';
        
        // Add a message about saving to validate
        echo '<div id="csp-validation-message" style="background: #e7f3ff; color: #0066cc; padding: 8px; border-radius: 4px; margin: 10px 0; border-left: 4px solid #0066cc; display: none;">';
        echo '<strong>💡 Tip:</strong> Make changes to the CSP policy above, then click "Save Changes" to validate your policy.';
        echo '</div>';
        
        // Display validation results
        if (!$validation['valid']) {
            echo '<div style="background: #f8d7da; color: #721c24; padding: 10px; border-radius: 4px; margin: 10px 0; border-left: 4px solid #dc3545;">';
            echo '<strong>❌ CSP Policy Validation Errors:</strong><ul style="margin: 5px 0 0 20px;">';
            foreach ($validation['errors'] as $error) {
                echo '<li>' . esc_html($error) . '</li>';
            }
            echo '</ul></div>';
        } elseif (!empty($validation['warnings'])) {
            echo '<div style="background: #fff3cd; color: #856404; padding: 10px; border-radius: 4px; margin: 10px 0; border-left: 4px solid #ffc107;">';
            echo '<strong>⚠️ CSP Policy Warnings:</strong><ul style="margin: 5px 0 0 20px;">';
            foreach ($validation['warnings'] as $warning) {
                echo '<li>' . esc_html($warning) . '</li>';
            }
            echo '</ul></div>';
        } else {
            echo '<div style="background: #d4edda; color: #155724; padding: 10px; border-radius: 4px; margin: 10px 0; border-left: 4px solid #28a745;">';
            echo '<strong>✅ CSP Policy looks good!</strong> No validation errors detected.';
            echo '</div>';
        }
        
        // Smart marketing based on user status
        $this->display_csp_builder_marketing();
    }
    
    public function csp_mode_callback() {
        $value = isset($this->options['csp_mode']) ? $this->options['csp_mode'] : 'enforcing';
        echo '<label><input type="radio" name="metricpoints_csp_manager_options[csp_mode]" value="enforcing" ' . checked('enforcing', $value, false) . ' /> Enforcing (blocks violations)</label><br>';
        echo '<label><input type="radio" name="metricpoints_csp_manager_options[csp_mode]" value="report-only" ' . checked('report-only', $value, false) . ' /> Report-Only (monitors violations without blocking)</label>';
        echo '<p class="description">Choose whether to enforce the CSP policy or just report violations for testing.</p>';
    }
    
    
    public function hsts_callback() {
        $value = isset($this->options['hsts_enabled']) ? $this->options['hsts_enabled'] : 0;
        $max_age = isset($this->options['hsts_max_age']) ? $this->options['hsts_max_age'] : 31536000;
        $include_subdomains = isset($this->options['hsts_include_subdomains']) ? $this->options['hsts_include_subdomains'] : 0;
        $preload = isset($this->options['hsts_preload']) ? $this->options['hsts_preload'] : 0;
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #28a745;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[hsts_enabled]" value="1" ' . checked(1, $value, false) . ' /> <strong>Enable HSTS (HTTP Strict Transport Security)</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Forces browsers to use HTTPS connections and prevents downgrade attacks</p>';
        
        echo '<div style="margin: 10px 0;">';
        echo '<label>Max Age: <input type="number" name="metricpoints_csp_manager_options[hsts_max_age]" value="' . esc_attr($max_age) . '" class="small-text" min="0" /> seconds</label>';
        echo '<span style="color: #666; font-size: 12px; margin-left: 10px;">(1 year = 31,536,000 seconds)</span>';
        echo '</div>';
        
        echo '<div style="margin: 10px 0;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[hsts_include_subdomains]" value="1" ' . checked(1, $include_subdomains, false) . ' /> Include subdomains</label>';
        echo '<span style="color: #666; font-size: 12px; margin-left: 10px;">Applies HSTS to all subdomains</span>';
        echo '</div>';
        
        echo '<div style="margin: 10px 0;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[hsts_preload]" value="1" ' . checked(1, $preload, false) . ' /> Preload</label>';
        echo '<span style="color: #666; font-size: 12px; margin-left: 10px;">Include in browser HSTS preload list</span>';
        echo '</div>';
        echo '</div>';
    }
    
    public function x_frame_options_callback() {
        $value = isset($this->options['x_frame_options']) ? $this->options['x_frame_options'] : 'SAMEORIGIN';
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #dc3545;">';
        echo '<label><strong>X-Frame-Options</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Protects against clickjacking attacks by controlling if your site can be embedded in frames</p>';
        
        echo '<select name="metricpoints_csp_manager_options[x_frame_options]" style="margin: 10px 0;">';
        echo '<option value=""' . selected('', $value, false) . '>Disabled - Do not set X-Frame-Options header</option>';
        echo '<option value="DENY"' . selected('DENY', $value, false) . '>DENY - Block all framing (most secure)</option>';
        echo '<option value="SAMEORIGIN"' . selected('SAMEORIGIN', $value, false) . '>SAMEORIGIN - Allow framing from same domain only (recommended)</option>';
        echo '<option value="ALLOW-FROM"' . selected('ALLOW-FROM', $value, false) . '>ALLOW-FROM - Allow framing from specific domains (deprecated)</option>';
        echo '</select>';
        
        echo '<div style="margin: 10px 0; padding: 8px; background: #e9ecef; border-radius: 3px; font-size: 12px;">';
        echo '<strong>💡 Recommendation:</strong> Use SAMEORIGIN for most websites. DENY only if you never need embedding.';
        echo '</div>';
        echo '</div>';
    }
    
    public function x_content_type_options_callback() {
        $value = isset($this->options['x_content_type_options']) ? $this->options['x_content_type_options'] : 1;
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #ffc107;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[x_content_type_options]" value="1" ' . checked(1, $value, false) . ' /> <strong>X-Content-Type-Options: nosniff</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Prevents browsers from MIME-sniffing responses and forces them to respect declared content types</p>';
        
        echo '<div style="margin: 10px 0; padding: 8px; background: #fff3cd; border-radius: 3px; font-size: 12px;">';
        echo '<strong>🛡️ Security Benefit:</strong> Prevents malicious files from being executed by tricking browsers into treating them as different file types.';
        echo '</div>';
        echo '</div>';
    }
    
    public function referrer_policy_callback() {
        $value = isset($this->options['referrer_policy']) ? $this->options['referrer_policy'] : 'strict-origin-when-cross-origin';
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #6f42c1;">';
        echo '<label><strong>Referrer-Policy</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Controls how much referrer information is sent with requests to protect user privacy</p>';
        
        echo '<select name="metricpoints_csp_manager_options[referrer_policy]" style="margin: 10px 0; width: 100%; max-width: 400px;">';
        echo '<option value=""' . selected('', $value, false) . '>Disabled - Do not set Referrer-Policy header</option>';
        echo '<option value="no-referrer"' . selected('no-referrer', $value, false) . '>no-referrer - Never send referrer information</option>';
        echo '<option value="no-referrer-when-downgrade"' . selected('no-referrer-when-downgrade', $value, false) . '>no-referrer-when-downgrade - Send referrer only for same-origin requests</option>';
        echo '<option value="origin"' . selected('origin', $value, false) . '>origin - Send only the origin (domain) as referrer</option>';
        echo '<option value="origin-when-cross-origin"' . selected('origin-when-cross-origin', $value, false) . '>origin-when-cross-origin - Send full referrer for same-origin, origin only for cross-origin</option>';
        echo '<option value="same-origin"' . selected('same-origin', $value, false) . '>same-origin - Send referrer only for same-origin requests</option>';
        echo '<option value="strict-origin"' . selected('strict-origin', $value, false) . '>strict-origin - Send only origin, never full URL</option>';
        echo '<option value="strict-origin-when-cross-origin"' . selected('strict-origin-when-cross-origin', $value, false) . '>strict-origin-when-cross-origin - Recommended default (privacy-focused)</option>';
        echo '<option value="unsafe-url"' . selected('unsafe-url', $value, false) . '>unsafe-url - Always send full referrer (least private)</option>';
        echo '</select>';
        
        echo '<div style="margin: 10px 0; padding: 8px; background: #e2e3f1; border-radius: 3px; font-size: 12px;">';
        echo '<strong>🔒 Privacy Note:</strong> "strict-origin-when-cross-origin" is recommended as it balances functionality with privacy protection.';
        echo '</div>';
        echo '</div>';
    }
    
    public function x_xss_protection_callback() {
        $value = isset($this->options['x_xss_protection']) ? $this->options['x_xss_protection'] : '1; mode=block';
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #dc3545;">';
        echo '<label><strong>X-XSS-Protection</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Enables the XSS filter built into most browsers. Note: Modern browsers rely on CSP instead, but this provides additional protection for older browsers.</p>';
        
        echo '<select name="metricpoints_csp_manager_options[x_xss_protection]" style="margin: 10px 0;">';
        echo '<option value=""' . selected('', $value, false) . '>Disabled - Do not set X-XSS-Protection header</option>';
        echo '<option value="0"' . selected('0', $value, false) . '>0 - Disable XSS filtering</option>';
        echo '<option value="1"' . selected('1', $value, false) . '>1 - Enable XSS filtering</option>';
        echo '<option value="1; mode=block"' . selected('1; mode=block', $value, false) . '>1; mode=block - Enable and block page rendering if attack detected (recommended)</option>';
        echo '</select>';
        
        echo '<div style="margin: 10px 0; padding: 8px; background: #e9ecef; border-radius: 3px; font-size: 12px;">';
        echo '<strong>💡 Note:</strong> Modern browsers (Chrome, Edge) ignore this header in favor of CSP. Still useful for older browsers.';
        echo '</div>';
        echo '</div>';
    }
    
    public function permissions_policy_callback() {
        $enabled = isset($this->options['permissions_policy_enabled']) ? $this->options['permissions_policy_enabled'] : 0;
        $features = isset($this->options['permissions_policy_features']) ? $this->options['permissions_policy_features'] : array();
        
        $available_features = array(
            'accelerometer' => 'Accelerometer',
            'autoplay' => 'Autoplay',
            'camera' => 'Camera',
            'encrypted-media' => 'Encrypted Media',
            'fullscreen' => 'Fullscreen',
            'geolocation' => 'Geolocation',
            'gyroscope' => 'Gyroscope',
            'magnetometer' => 'Magnetometer',
            'microphone' => 'Microphone',
            'midi' => 'MIDI',
            'payment' => 'Payment',
            'picture-in-picture' => 'Picture-in-Picture',
            'sync-xhr' => 'Synchronous XHR',
            'usb' => 'USB',
            'xr-spatial-tracking' => 'XR Spatial Tracking',
        );
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #6f42c1;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[permissions_policy_enabled]" value="1" ' . checked(1, $enabled, false) . ' /> <strong>Enable Permissions-Policy</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Control which browser features and APIs your site can use. Restrict features to improve security and privacy.</p>';
        
        $features_style = $enabled ? 'display: grid;' : 'display: none;';
        echo '<div style="margin: 10px 0; padding: 10px; background: #fff; border: 1px solid #ddd; border-radius: 3px;" id="permissions-policy-features-container">';
        echo '<p style="margin: 0 0 10px 0; font-weight: bold;">Feature Permissions:</p>';
        echo '<div style="' . $features_style . ' grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 10px;" id="permissions-policy-features-grid">';
            
        foreach ($available_features as $feature_key => $feature_name) {
            $feature_value = isset($features[$feature_key]) ? $features[$feature_key] : 'self';
            echo '<div style="padding: 8px; background: #f8f9fa; border-radius: 3px;">';
            echo '<label style="font-weight: bold; display: block; margin-bottom: 5px;">' . esc_html($feature_name) . '</label>';
            echo '<select name="metricpoints_csp_manager_options[permissions_policy_features][' . esc_attr($feature_key) . ']" style="width: 100%;">';
            echo '<option value="*"' . selected('*', $feature_value, false) . '>Allow all</option>';
            echo '<option value="self"' . selected('self', $feature_value, false) . '>Same origin only</option>';
            echo '<option value="none"' . selected('none', $feature_value, false) . '>Block all</option>';
            echo '</select>';
            echo '</div>';
        }
            
        echo '</div>';
        echo '<p style="margin: 10px 0 0 0; font-size: 12px; color: #666;">💡 <strong>Tip:</strong> Use "Block all" for features you don\'t need to improve security. Use "Same origin only" for features you need but want to restrict.</p>';
        echo '</div>';
        echo '</div>'; // Close outer div opened on line 522
    }
    
    public function expect_ct_callback() {
        $enabled = isset($this->options['expect_ct_enabled']) ? $this->options['expect_ct_enabled'] : 0;
        $max_age = isset($this->options['expect_ct_max_age']) ? $this->options['expect_ct_max_age'] : 86400;
        $enforce = isset($this->options['expect_ct_enforce']) ? $this->options['expect_ct_enforce'] : 0;
        $report_uri = isset($this->options['expect_ct_report_uri']) ? $this->options['expect_ct_report_uri'] : '';
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #17a2b8;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[expect_ct_enabled]" value="1" ' . checked(1, $enabled, false) . ' /> <strong>Enable Expect-CT</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Certificate Transparency (CT) helps detect misissued SSL certificates. Note: This header is deprecated but still supported by some browsers.</p>';
        
        $expect_ct_style = $enabled ? 'display: block;' : 'display: none;';
        echo '<div style="margin: 10px 0; ' . $expect_ct_style . '" id="expect-ct-settings">';
        echo '<label>Max Age: <input type="number" name="metricpoints_csp_manager_options[expect_ct_max_age]" value="' . esc_attr($max_age) . '" class="small-text" min="0" /> seconds</label>';
        echo '<span style="color: #666; font-size: 12px; margin-left: 10px;">(1 day = 86,400 seconds, 1 year = 31,536,000 seconds)</span>';
        echo '</div>';
        
        echo '<div style="margin: 10px 0; ' . $expect_ct_style . '" id="expect-ct-enforce">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[expect_ct_enforce]" value="1" ' . checked(1, $enforce, false) . ' /> Enforce (reject connections without valid CT)</label>';
        echo '</div>';
        
        echo '<div style="margin: 10px 0; ' . $expect_ct_style . '" id="expect-ct-report-uri">';
        echo '<label>Report URI (optional): <input type="url" name="metricpoints_csp_manager_options[expect_ct_report_uri]" value="' . esc_attr($report_uri) . '" class="regular-text" placeholder="https://example.com/report-ct" /></label>';
        echo '</div>';
        
        echo '<div style="margin: 10px 0; padding: 8px; background: #d1ecf1; border-radius: 3px; font-size: 12px;">';
        echo '<strong>⚠️ Note:</strong> Expect-CT is deprecated. Modern browsers use Certificate Transparency logs built-in.';
        echo '</div>';
        echo '</div>';
    }
    
    public function clear_site_data_callback() {
        $enabled = isset($this->options['clear_site_data_enabled']) ? $this->options['clear_site_data_enabled'] : 0;
        $directives = isset($this->options['clear_site_data_directives']) ? $this->options['clear_site_data_directives'] : array();
        
        $available_directives = array(
            'cache' => 'Cache - Clear cached resources',
            'cookies' => 'Cookies - Clear all cookies',
            'storage' => 'Storage - Clear localStorage, sessionStorage, etc.',
            'executionContexts' => 'Execution Contexts - Clear service workers and shared workers',
        );
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #ffc107;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[clear_site_data_enabled]" value="1" ' . checked(1, $enabled, false) . ' /> <strong>Enable Clear-Site-Data</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Instructs browsers to clear specified types of data when the header is received. Useful for logout pages or data cleanup.</p>';
        
        $clear_site_data_style = $enabled ? 'display: block;' : 'display: none;';
        echo '<div style="margin: 10px 0; padding: 10px; background: #fff; border: 1px solid #ddd; border-radius: 3px; ' . $clear_site_data_style . '" id="clear-site-data-settings">';
        echo '<p style="margin: 0 0 10px 0; font-weight: bold;">Clear Data Types:</p>';
        
        foreach ($available_directives as $directive_key => $directive_label) {
            $checked = isset($directives[$directive_key]) && $directives[$directive_key] ? 'checked' : '';
            echo '<label style="display: block; margin: 5px 0;">';
            echo '<input type="checkbox" name="metricpoints_csp_manager_options[clear_site_data_directives][' . esc_attr($directive_key) . ']" value="1" ' . $checked . ' /> ';
            echo esc_html($directive_label);
            echo '</label>';
        }
        
        echo '<p style="margin: 10px 0 0 0; font-size: 12px; color: #666;">💡 <strong>Warning:</strong> This header clears data immediately when sent. Use carefully!</p>';
        echo '</div>';
        
        echo '</div>';
    }
    
    public function cross_origin_headers_callback() {
        $coep_enabled = isset($this->options['coep_enabled']) ? $this->options['coep_enabled'] : 0;
        $coep_value = isset($this->options['coep_value']) ? $this->options['coep_value'] : 'require-corp';
        $coop_enabled = isset($this->options['coop_enabled']) ? $this->options['coop_enabled'] : 0;
        $coop_value = isset($this->options['coop_value']) ? $this->options['coop_value'] : 'same-origin';
        $corp_enabled = isset($this->options['corp_enabled']) ? $this->options['corp_enabled'] : 0;
        $corp_value = isset($this->options['corp_value']) ? $this->options['corp_value'] : 'same-origin';
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #20c997;">';
        echo '<label><strong>Cross-Origin Headers</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Control how resources from different origins can be embedded and accessed. Important for security isolation.</p>';
        
        // COEP
        $coep_style = $coep_enabled ? 'display: block;' : 'display: none;';
        echo '<div style="margin: 10px 0; padding: 10px; background: #fff; border: 1px solid #ddd; border-radius: 3px;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[coep_enabled]" value="1" ' . checked(1, $coep_enabled, false) . ' /> <strong>Cross-Origin-Embedder-Policy (COEP)</strong></label><br>';
        echo '<select name="metricpoints_csp_manager_options[coep_value]" style="margin: 5px 0; ' . $coep_style . '" id="coep-select">';
        echo '<option value="unsafe-none"' . selected('unsafe-none', $coep_value, false) . '>unsafe-none - Allow all cross-origin embeds (least secure)</option>';
        echo '<option value="require-corp"' . selected('require-corp', $coep_value, false) . '>require-corp - Require CORP header for cross-origin resources (recommended)</option>';
        echo '</select>';
        echo '<p style="margin: 5px 0 0 0; font-size: 12px; color: #666; ' . $coep_style . '" id="coep-description">Prevents your site from loading cross-origin resources that don\'t explicitly grant permission.</p>';
        echo '</div>';
        
        // COOP
        $coop_style = $coop_enabled ? 'display: block;' : 'display: none;';
        echo '<div style="margin: 10px 0; padding: 10px; background: #fff; border: 1px solid #ddd; border-radius: 3px;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[coop_enabled]" value="1" ' . checked(1, $coop_enabled, false) . ' /> <strong>Cross-Origin-Opener-Policy (COOP)</strong></label><br>';
        echo '<select name="metricpoints_csp_manager_options[coop_value]" style="margin: 5px 0; ' . $coop_style . '" id="coop-select">';
        echo '<option value="unsafe-none"' . selected('unsafe-none', $coop_value, false) . '>unsafe-none - Allow cross-origin window access (least secure)</option>';
        echo '<option value="same-origin-allow-popups"' . selected('same-origin-allow-popups', $coop_value, false) . '>same-origin-allow-popups - Isolate but allow popups</option>';
        echo '<option value="same-origin"' . selected('same-origin', $coop_value, false) . '>same-origin - Isolate from cross-origin windows (most secure)</option>';
        echo '</select>';
        echo '<p style="margin: 5px 0 0 0; font-size: 12px; color: #666; ' . $coop_style . '" id="coop-description">Isolates your browsing context from cross-origin documents.</p>';
        echo '</div>';
        
        // CORP
        $corp_style = $corp_enabled ? 'display: block;' : 'display: none;';
        echo '<div style="margin: 10px 0; padding: 10px; background: #fff; border: 1px solid #ddd; border-radius: 3px;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[corp_enabled]" value="1" ' . checked(1, $corp_enabled, false) . ' /> <strong>Cross-Origin-Resource-Policy (CORP)</strong></label><br>';
        echo '<select name="metricpoints_csp_manager_options[corp_value]" style="margin: 5px 0; ' . $corp_style . '" id="corp-select">';
        echo '<option value="same-site"' . selected('same-site', $corp_value, false) . '>same-site - Allow same-site requests only</option>';
        echo '<option value="same-origin"' . selected('same-origin', $corp_value, false) . '>same-origin - Allow same-origin requests only (most secure)</option>';
        echo '<option value="cross-origin"' . selected('cross-origin', $corp_value, false) . '>cross-origin - Allow cross-origin requests</option>';
        echo '</select>';
        echo '<p style="margin: 5px 0 0 0; font-size: 12px; color: #666; ' . $corp_style . '" id="corp-description">Controls which sites can load your resources.</p>';
        echo '</div>';
        
        echo '<div style="margin: 10px 0; padding: 8px; background: #d1ecf1; border-radius: 3px; font-size: 12px;">';
        echo '<strong>💡 Note:</strong> These headers work together. COEP requires CORP on cross-origin resources. Test thoroughly!';
        echo '</div>';
        echo '</div>';
    }
    
    public function custom_headers_callback() {
        $value = isset($this->options['custom_headers']) ? $this->options['custom_headers'] : '';
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #6c757d;">';
        echo '<label><strong>Custom Headers</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Add any additional security headers you need. One header per line in the format: <code>Header-Name: Header-Value</code></p>';
        
        echo '<textarea name="metricpoints_csp_manager_options[custom_headers]" id="custom-headers-textarea" rows="6" cols="50" class="large-text" placeholder="X-Custom-Header: value&#10;Another-Header: another-value">' . esc_textarea($value) . '</textarea>';
        
        echo '<div style="margin: 10px 0; padding: 8px; background: #e9ecef; border-radius: 3px; font-size: 12px;">';
        echo '<strong>💡 Examples:</strong><br>';
        echo '<code>X-Custom-Header: my-value</code><br>';
        echo '<code>Feature-Policy: geolocation \'none\'</code>';
        echo '</div>';
        echo '</div>';
    }
    
    public function admin_headers_callback() {
        $value = isset($this->options['admin_headers']) ? $this->options['admin_headers'] : 0;
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #17a2b8;">';
        echo '<label><input type="checkbox" name="metricpoints_csp_manager_options[admin_headers]" value="1" ' . checked(1, $value, false) . ' /> <strong>Enable Headers in WordPress Admin</strong></label><br>';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Apply CSP and security headers to WordPress admin pages (wp-admin). Useful for testing and securing admin area.</p>';
        
        echo '<div style="margin: 10px 0; padding: 8px; background: #d1ecf1; border-radius: 3px; font-size: 12px;">';
        echo '<strong>⚠️ Note:</strong> Enabling this may break some WordPress admin functionality if your CSP policy is too restrictive. Test carefully!';
        echo '</div>';
        echo '</div>';
    }
    
    public function backup_limit_callback() {
        $value = isset($this->options['backup_limit']) ? $this->options['backup_limit'] : 20;
        $current_backups = get_option('metricpoints_csp_policy_backups', array());
        $backup_count = count($current_backups);
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #17a2b8;">';
        echo '<label for="backup_limit"><strong>Maximum Policy Backups to Keep</strong></label><br>';
        echo '<input type="number" id="backup_limit" name="metricpoints_csp_manager_options[backup_limit]" value="' . esc_attr($value) . '" min="10" max="100" style="width: 80px; margin: 5px 0;" />';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Number of policy backups to maintain (minimum: 10, maximum: 100). Older backups are automatically removed when this limit is exceeded.</p>';
        
        echo '<div style="margin: 10px 0; padding: 8px; background: #d1ecf1; border-radius: 3px; font-size: 12px;">';
        echo '<strong>📊 Current Status:</strong> ' . $backup_count . ' backups stored (limit: ' . $value . ')';
        if ($backup_count >= $value) {
            echo ' <span style="color: #856404;">⚠️ At limit - oldest backup will be removed on next policy update</span>';
        }
        echo '</div>';
        echo '</div>';
    }
    
    public function confidence_threshold_callback() {
        $value = isset($this->options['confidence_threshold']) ? $this->options['confidence_threshold'] : 80;
        $recent_updates = get_option('metricpoints_csp_policy_updates', array());
        $recent_updates = array_slice($recent_updates, -5); // Last 5 updates
        
        echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 4px solid #17a2b8;">';
        echo '<label for="confidence_threshold"><strong>Minimum Confidence for Automatic Updates</strong></label><br>';
        echo '<input type="number" id="confidence_threshold" name="metricpoints_csp_manager_options[confidence_threshold]" value="' . esc_attr($value) . '" min="50" max="100" step="5" style="width: 80px; margin: 5px 0;" />%';
        echo '<p style="margin: 5px 0 10px 0; color: #666; font-size: 13px;">Policy updates below this confidence level will be skipped and require manual activation. Updates at or above this level will be applied automatically.</p>';
        
        if (!empty($recent_updates)) {
            echo '<div style="margin: 10px 0; padding: 8px; background: #d1ecf1; border-radius: 3px; font-size: 12px;">';
            echo '<strong>📊 Recent Updates:</strong><br>';
            foreach ($recent_updates as $update) {
                $confidence = isset($update['confidence_score']) ? round($update['confidence_score'] * 100) : 0;
                $status = $confidence >= $value ? '✅ Applied' : '⏸️ Skipped';
                $date = isset($update['timestamp']) ? date('M j, H:i', strtotime($update['timestamp'])) : 'Unknown';
                echo '<span style="margin-right: 15px;">' . $date . ': ' . $confidence . '% ' . $status . '</span>';
            }
            echo '</div>';
        }
        echo '</div>';
    }
    
    public function validate_options($input) {
        $validated = array();
        
        // Validate backup limit
        if (isset($input['backup_limit'])) {
            $backup_limit = intval($input['backup_limit']);
            if ($backup_limit < 10) {
                $backup_limit = 10;
                add_settings_error('metricpoints_csp_manager_options', 'backup_limit_min', 'Backup limit cannot be less than 10. Set to 10.');
            } elseif ($backup_limit > 100) {
                $backup_limit = 100;
                add_settings_error('metricpoints_csp_manager_options', 'backup_limit_max', 'Backup limit cannot be more than 100. Set to 100.');
            }
            $validated['backup_limit'] = $backup_limit;
        }
        
        // Validate confidence threshold
        if (isset($input['confidence_threshold'])) {
            $confidence_threshold = intval($input['confidence_threshold']);
            if ($confidence_threshold < 50) {
                $confidence_threshold = 50;
                add_settings_error('metricpoints_csp_manager_options', 'confidence_threshold_min', 'Confidence threshold cannot be less than 50%. Set to 50%.');
            } elseif ($confidence_threshold > 100) {
                $confidence_threshold = 100;
                add_settings_error('metricpoints_csp_manager_options', 'confidence_threshold_max', 'Confidence threshold cannot be more than 100%. Set to 100%.');
            }
            $validated['confidence_threshold'] = $confidence_threshold;
        }
        
        // Validate and verify API key if provided
        if (isset($input['api_key']) && !empty($input['api_key'])) {
            $api_key = sanitize_text_field($input['api_key']);
            
            // Verify API key with MetricPoints
            $verification = $this->verify_api_key($api_key);
            
            if ($verification['success']) {
                $validated['api_key'] = $api_key;
                // Store project info for display
                $validated['project_info'] = $verification['project'] ?? null;
                $validated['connection_status'] = 'connected';
                $validated['connection_verified_at'] = current_time('mysql');
                
                // Add success message
                add_settings_error('metricpoints_csp_manager_options', 'api_key_valid', '✅ API key verified successfully! Connected to project: ' . ($verification['project']['name'] ?? 'Unknown'), 'updated');
            } else {
                // Keep old API key if verification fails
                $old_options = get_option('metricpoints_csp_manager_options', array());
                $validated['api_key'] = $old_options['api_key'] ?? '';
                $validated['connection_status'] = 'failed';
                
                add_settings_error('metricpoints_csp_manager_options', 'api_key_invalid', '❌ API key verification failed: ' . ($verification['error'] ?? 'Invalid API key. Please check your API key and try again.'), 'error');
            }
        } elseif (isset($input['api_key']) && empty($input['api_key'])) {
            // API key cleared
            $validated['api_key'] = '';
            $validated['connection_status'] = 'disconnected';
            $validated['project_info'] = null;
        } else {
            // Keep existing API key
            $old_options = get_option('metricpoints_csp_manager_options', array());
            $validated['api_key'] = $old_options['api_key'] ?? '';
            $validated['project_info'] = $old_options['project_info'] ?? null;
            $validated['connection_status'] = $old_options['connection_status'] ?? 'unknown';
        }
        
        // Sync manual policy update if CSP policy changed
        if (isset($input['csp_policy'])) {
            $old_options = get_option('metricpoints_csp_manager_options', array());
            $old_policy = $old_options['csp_policy'] ?? '';
            $new_policy = $input['csp_policy'];
            
            if (!empty($validated['api_key']) && $old_policy !== $new_policy && !empty($old_policy)) {
                // Policy was manually changed - sync to MetricPoints
                $this->sync_policy_update_to_metricpoints($old_policy, $new_policy, 'manual');
            }
            
            $validated['csp_policy'] = $new_policy;
        }
        
        // Validate other options (pass through for now)
        foreach ($input as $key => $value) {
            if (!in_array($key, ['backup_limit', 'confidence_threshold', 'api_key', 'csp_policy', 'project_info', 'connection_status', 'connection_verified_at'])) {
                $validated[$key] = $value;
            }
        }
        
        return $validated;
    }
    
    public function admin_page() {
        ?>
        <div class="wrap">
            <h1>MetricPoints CSP Manager</h1>
            
            <!-- Connection Status Card -->
            <div class="card" style="margin-bottom: 20px;">
                <h2>🔗 Connection Status</h2>
                <?php 
                $connection_status = $this->options['connection_status'] ?? 'unknown';
                $api_key = $this->options['api_key'] ?? '';
                $project_info = $this->options['project_info'] ?? null;
                
                if (!empty($api_key)):
                    if ($connection_status === 'connected' && $project_info):
                        ?>
                        <div style="background: #d4edda; padding: 15px; border-radius: 5px; border-left: 4px solid #28a745; margin: 10px 0;">
                            <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 10px;">
                                <span style="display: inline-block; width: 12px; height: 12px; background: #28a745; border-radius: 50%;"></span>
                                <p style="margin: 0; color: #155724;"><strong>✅ Connected to MetricPoints</strong></p>
                                <button id="test-connection-btn" class="button button-small" style="margin-left: auto;">🔄 Test Connection</button>
                            </div>
                            <p style="margin: 5px 0 0 0; color: #155724;">
                                <strong>Project:</strong> <?php echo esc_html($project_info['name'] ?? 'Unknown'); ?><br>
                                <strong>Domain:</strong> <?php echo esc_html($project_info['domain'] ?? 'Unknown'); ?><br>
                                <?php if (isset($this->options['connection_verified_at'])): ?>
                                    <strong>Last Verified:</strong> <?php echo esc_html(date('M j, Y g:i A', strtotime($this->options['connection_verified_at']))); ?>
                                <?php endif; ?>
                            </p>
                            <div id="test-connection-result" style="margin-top: 10px;"></div>
                        </div>
                        <?php
                    else:
                        ?>
                        <div style="background: #fff3cd; padding: 15px; border-radius: 5px; border-left: 4px solid #ffc107; margin: 10px 0;">
                            <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 10px;">
                                <span style="display: inline-block; width: 12px; height: 12px; background: #ffc107; border-radius: 50%;"></span>
                                <p style="margin: 0; color: #856404;"><strong>⚠️ Connection Status Unknown</strong></p>
                            </div>
                            <p style="margin: 5px 0 0 0; color: #856404;">API key is set but connection status is not verified. Click "Verify Connection" to check.</p>
                            <button id="verify-connection-btn" class="button" style="margin-top: 10px;">🔄 Verify Connection</button>
                            <div id="verify-connection-result" style="margin-top: 10px;"></div>
                        </div>
                        <?php
                    endif;
                else:
                    ?>
                    <div style="background: #f8d7da; padding: 15px; border-radius: 5px; border-left: 4px solid #dc3545; margin: 10px 0;">
                        <p style="margin: 0; color: #721c24;"><strong>❌ Not Connected</strong></p>
                        <p style="margin: 5px 0 0 0; color: #721c24;">Please enter your MetricPoints API key below to connect your account.</p>
                        <div style="margin-top: 15px;">
                            <button type="button" id="metricpoints-register-btn" class="button button-primary" style="margin-right: 10px;">
                                🚀 Create MetricPoints Account
                            </button>
                            <a href="https://metricpoints.com" target="_blank" class="button button-secondary">
                                Learn More
                            </a>
                        </div>
                        <div id="metricpoints-registration-form" style="display: none; margin-top: 15px; padding: 15px; background: #f8f9fa; border-radius: 5px;">
                            <h3 style="margin-top: 0;">Create Your MetricPoints Account</h3>
                            <form id="metricpoints-register-form-form">
                                <table class="form-table">
                                    <tr>
                                        <th scope="row"><label for="reg-name">Name</label></th>
                                        <td><input type="text" id="reg-name" name="name" class="regular-text" required /></td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="reg-email">Email</label></th>
                                        <td><input type="email" id="reg-email" name="email" class="regular-text" required /></td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="reg-password">Password</label></th>
                                        <td><input type="password" id="reg-password" name="password" class="regular-text" required minlength="8" /></td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="reg-domain">Domain (optional)</label></th>
                                        <td><input type="text" id="reg-domain" name="domain" class="regular-text" value="<?php echo esc_attr(parse_url(home_url(), PHP_URL_HOST)); ?>" /></td>
                                    </tr>
                                </table>
                                <p class="submit">
                                    <button type="submit" class="button button-primary">Create Account & Connect</button>
                                    <button type="button" id="metricpoints-cancel-register" class="button">Cancel</button>
                                </p>
                                <div id="metricpoints-register-result" style="margin-top: 10px;"></div>
                            </form>
                        </div>
                    </div>
                    <?php
                endif;
                ?>
            </div>
            
            <div class="card">
                <h2>Status</h2>
                <?php if (!empty($this->options['enabled'])): ?>
                    <p style="color: green;">✅ CSP management is active</p>
                    <p><strong>Current Policy:</strong></p>
                    <div style="background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto;">
                        <pre style="margin: 0; white-space: pre-wrap; word-wrap: break-word; min-width: 0;"><?php echo esc_html($this->get_current_csp_policy()); ?></pre>
                    </div>
                    
                    <?php if (!empty($this->options['custom_headers'])): ?>
                        <p style="margin-top: 15px;"><strong>Custom Headers:</strong></p>
                        <div style="background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto;">
                            <pre style="margin: 0; white-space: pre-wrap; word-wrap: break-word; min-width: 0;"><?php echo esc_html($this->options['custom_headers']); ?></pre>
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <p style="color: red;">❌ CSP management is disabled</p>
                <?php endif; ?>
            </div>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('metricpoints_csp_manager_options');
                ?>
                
                <!-- Basic Settings -->
                <table class="form-table">
                    <tbody>
                        <tr>
                            <th scope="row">API Key</th>
                            <td><?php $this->api_key_callback(); ?></td>
                        </tr>
                        <tr>
                            <th scope="row">Enable CSP Management</th>
                            <td><?php $this->enabled_callback(); ?></td>
                        </tr>
                        <tr>
                            <th scope="row">Enable Self-Healing</th>
                            <td><?php $this->self_healing_callback(); ?></td>
                        </tr>
                        <tr>
                            <th scope="row">CSP Policy</th>
                            <td><?php $this->csp_policy_callback(); ?></td>
                        </tr>
                        <tr>
                            <th scope="row">CSP Mode</th>
                            <td><?php $this->csp_mode_callback(); ?></td>
                        </tr>
                    </tbody>
                </table>
                
                <!-- Security Headers Section (Collapsible) -->
                <div style="margin: 20px 0;">
                    <h2>🛡️ Additional Headers and Advanced Settings<span id="security-headers-toggle" style="cursor: pointer; color: #0073aa; font-size: 16px;">▼</span></h2>
                    <p style="color: #666; margin: 5px 0 0 0;">Configure additional headers and advanced settings to enhance your website's security posture</p>
                </div>
                
                <div id="security-headers-settings" style="display: none;">
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row">Strict-Transport-Security (HSTS)</th>
                                <td><?php $this->hsts_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">X-Frame-Options</th>
                                <td><?php $this->x_frame_options_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">X-Content-Type-Options</th>
                                <td><?php $this->x_content_type_options_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Referrer-Policy</th>
                                <td><?php $this->referrer_policy_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">X-XSS-Protection</th>
                                <td><?php $this->x_xss_protection_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Permissions-Policy</th>
                                <td><?php $this->permissions_policy_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Expect-CT</th>
                                <td><?php $this->expect_ct_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Clear-Site-Data</th>
                                <td><?php $this->clear_site_data_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Cross-Origin Headers</th>
                                <td><?php $this->cross_origin_headers_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Custom Headers</th>
                                <td><?php $this->custom_headers_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Enable Headers in Admin</th>
                                <td><?php $this->admin_headers_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Policy Backup Limit</th>
                                <td><?php $this->backup_limit_callback(); ?></td>
                            </tr>
                            <tr>
                                <th scope="row">Policy Update Confidence Threshold</th>
                                <td><?php $this->confidence_threshold_callback(); ?></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                
                <?php submit_button(); ?>
            </form>
            
            <style>
                /* Ensure cards below Save Changes button take full available width */
                .wrap > .card {
                    max-width: none !important;
                    width: 100% !important;
                    box-sizing: border-box;
                }
                /* Ensure form and form-table don't constrain width unnecessarily */
                .wrap form {
                    max-width: 100%;
                }
                .wrap .form-table {
                    max-width: 100%;
                    width: 100%;
                }
                /* Make sure cards after the form break out of any width constraints */
                .wrap form + .card,
                .wrap form + .card + .card,
                .wrap form + .card + .card + .card {
                    max-width: 100% !important;
                    width: 100% !important;
                    margin-left: 0 !important;
                    margin-right: 0 !important;
                }
            </style>
            
            <div class="card">
                <h2>CSP Violations</h2>
                <p>Recent CSP violations detected:</p>
                <?php $this->display_violations(); ?>
            </div>
            
            <div class="card">
                <h2>Test CSP</h2>
                <p>Click the button below to test CSP violations:</p>
                <?php if (!empty($this->options['api_key']) && !empty($this->options['enabled'])): ?>
                <button id="test-csp-btn" class="button button-primary">Generate CSP Violation</button>
                <?php else: ?>
                    <button id="test-csp-btn" class="button button-primary" disabled>Generate CSP Violation</button>
                    <p class="description" style="color: #d63638;">Please configure your API Key and enable CSP management before testing.</p>
                <?php endif; ?>
                <div id="test-result" style="margin-top: 10px;"></div>
            </div>
            
            <div class="card">
                <h2>🤖 AI-Powered Self-Healing CSP</h2>
                <?php if (!empty($this->options['self_healing'])): ?>
                    <div style="background: #e8f5e8; padding: 15px; border-radius: 5px; margin: 10px 0;">
                        <p style="margin: 0; color: #2d5a2d;"><strong>✅ Self-Healing Active</strong> - Your CSP policy is being automatically maintained!</p>
                    </div>
                    
                    <!-- Webhook Configuration Section -->
                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 15px 0; border-left: 4px solid #007cba;">
                        <h3 style="margin: 0 0 15px 0; color: #007cba;">🔗 Webhook Configuration</h3>
                        <p style="margin: 0 0 15px 0; color: #666;">To enable self-healing, you need to configure a webhook in your MetricPoints dashboard that will notify this WordPress site when CSP policy updates are available.</p>
                        
                        <div style="background: #e3f2fd; padding: 15px; border-radius: 5px; margin: 10px 0;">
                            <h4 style="margin: 0 0 10px 0; color: #1976d2;">📋 Setup Instructions:</h4>
                            <ol style="margin: 0; padding-left: 20px; color: #666;">
                                <li>Go to your <a href="https://metricpoints.com/dashboard" target="_blank" style="color: #0073aa; font-weight: bold;">MetricPoints Dashboard</a></li>
                                <li>Go to the CSP Sites page</li>
                                <li>Click on the "Manage Notifications" option under the site you want to add the webhook to</li>
                                <li>Under the "Add New Notification" option, select "MetricPoints WordPress Webhook"</li>
                                <li>Enter this URL into the Webhook URL (Make sure that the HTTP Method is set to POST):</li>
                            </ol>
                            
                            <div style="background: #fff; padding: 10px; border-radius: 4px; margin: 10px 0; border: 1px solid #ddd;">
                                <code style="background: #f5f5f5; padding: 5px; border-radius: 3px; font-family: monospace; word-break: break-all;">
                                    <?php echo home_url('/wp-json/metricpoints-csp/v1/webhook'); ?>
                                </code>
                                <br><br>
                                <button onclick="copyWebhookUrl()" style="margin-left: 10px; padding: 5px 10px; background: #0073aa; color: white; border: none; border-radius: 3px; cursor: pointer;">📋 Copy</button>
                            </div>
                            
                            <p style="margin: 10px 0 0 0; color: #666; font-size: 14px;">
                                <strong>💡 Tip:</strong> The webhook will be called whenever MetricPoints detects CSP violations and generates policy recommendations.
                            </p>
                        </div>
                        
                        <div style="background: #fff3cd; padding: 10px; border-radius: 4px; margin: 10px 0; border-left: 3px solid #ffc107;">
                            <strong>⚠️ Note:</strong> Make sure your WordPress site is publicly accessible for webhook delivery. If you're testing locally, consider using a service like ngrok to expose your local WordPress installation.
                        </div>
                    </div>
                    
                    <div style="margin: 15px 0;">
                        <p style="margin: 0 0 10px 0; color: #666; font-size: 13px;">
                            <strong>💡 Manual Analysis Tools:</strong> Self-healing works automatically, but you can manually trigger analysis or policy updates for testing and debugging purposes.
                        </p>
                        <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                            <button id="analyze-violations-btn" class="button button-primary" title="Manually analyze current violations and generate recommendations (for testing/debugging)">🧠 Analyze Violations & Learn</button>
                            <button id="force-update-btn" class="button" title="Force an immediate policy update with current recommendations">⚡ Force Policy Update</button>
                            <button id="reset-learning-btn" class="button" title="Clear all learned patterns and start fresh">🔄 Reset Learning Data</button>
                            <button id="rollback-btn" class="button" style="background: #dc3545; color: white;" title="Restore a previous version of your CSP policy">↩️ Rollback Policy</button>
                        </div>
                    </div>
                    
                    <div id="self-healing-status" style="margin-top: 15px;">
                        <h4>Self-Healing Status:</h4>
                        <div id="healing-stats">
                            <p><strong>Violations Analyzed:</strong> <span id="violations-count"><?php echo count($this->csp_violations); ?></span></p>
                            <p><strong>Policy Updates:</strong> <span id="updates-count"><?php echo $this->get_policy_update_count(); ?></span></p>
                            <p><strong>Last Update:</strong> <span id="last-update"><?php echo $this->get_last_policy_update(); ?></span></p>
                        </div>
                        
                        <div style="background: #f0f8ff; padding: 15px; border-radius: 5px; margin: 15px 0; border-left: 4px solid #007cba;">
                            <h4 style="margin: 0 0 10px 0;">🔗 Self-Healing Webhook</h4>
                            <p style="margin: 0 0 10px 0; color: #666;">When violations are detected, this webhook will be called to trigger policy analysis and updates:</p>
                            
                            <!-- Webhook Update History -->
                            <div style="margin-top: 15px;">
                                <h4 style="margin: 0 0 10px 0; color: #007cba;">📊 Recent Policy Updates</h4>
                                <?php $this->display_webhook_history(); ?>
                            </div>
                            <div style="background: #f8f9fa; padding: 10px; border-radius: 4px; font-family: monospace; word-break: break-all; border: 1px solid #dee2e6;">
                                <strong>Webhook URL:</strong><br>
                                <span id="webhook-url"><?php echo $this->get_self_healing_webhook_url(); ?></span>
                                <button onclick="copyWebhookUrl()" style="margin-left: 10px; padding: 4px 8px; font-size: 12px;" class="button button-small">📋 Copy</button>
                                <button onclick="testWebhook()" style="margin-left: 5px; padding: 4px 8px; font-size: 12px;" class="button button-small">🧪 Test</button>
                            </div>
                            <p style="margin: 10px 0 0 0; font-size: 12px; color: #666;">
                                <strong>Method:</strong> POST | 
                                <strong>Content-Type:</strong> application/json | 
                                <strong>Authentication:</strong> API Key in Authorization header
                            </p>
                        </div>
                    </div>
                    
                    <div style="background: #f0f8ff; padding: 15px; border-radius: 5px; margin: 15px 0;">
                        <h4>🎯 What Self-Healing Does:</h4>
                        <ul style="margin: 10px 0; padding-left: 20px;">
                            <li>Analyzes violation patterns and learns from them</li>
                            <li>Automatically adds trusted domains to appropriate directives</li>
                            <li>Removes unused or problematic sources</li>
                            <li>Optimizes directive ordering for better performance</li>
                            <li>Maintains security while reducing false positives</li>
                        </ul>
                    </div>
                    
                    <div style="background: #fff8e1; padding: 15px; border-radius: 5px; margin: 15px 0;">
                        <h4>🛡️ Policy Backups & Safety:</h4>
                        <p>Your CSP policy is automatically backed up before each AI update. You can rollback to any previous version if needed.</p>
                        <div id="policy-backups">
                            <?php $this->display_policy_backups(); ?>
                        </div>
                        
                        <!-- Backup Preview Area -->
                        <div id="backup-preview" style="margin-top: 15px; display: none;">
                            <!-- Preview content will be inserted here -->
                        </div>
                    </div>
                <?php else: ?>
                    <div style="background: #fff3cd; padding: 15px; border-radius: 5px; margin: 10px 0;">
                        <p style="margin: 0; color: #856404;"><strong>⚠️ Self-Healing Disabled</strong> - Enable it to experience the future of CSP management!</p>
                    </div>
                    <p>Self-healing is disabled. Enable it to automatically update your CSP policy based on real violation data.</p>
                <?php endif; ?>
            </div>
        </div>
        
        <script>
        if (typeof metricpointsCspManager === 'undefined') {
            // Define metricpointsCspManager inline
            window.metricpointsCspManager = {
                ajaxUrl: '<?php echo admin_url('admin-ajax.php'); ?>',
                nonce: '<?php echo wp_create_nonce('metricpoints_csp_manager_nonce'); ?>',
                reportUri: '<?php echo admin_url('admin-ajax.php?action=metricpoints_csp_violation'); ?>'
            };
        }
        
        // Auto-extract API key from pasted URL
        function extractApiKey(value) {
            // Handle various URL patterns for CSP reports:
            // - https://metricpoints.com/api/csp-reports/e3bcbc89-2846-48c1-a393-66b635478cce
            // - @https://metricpoints.com/api/csp-reports/e3bcbc89-2846-48c1-a393-66b635478cce
            // - metricpoints.com/api/csp-reports/e3bcbc89-2846-48c1-a393-66b635478cce
            // - /api/csp-reports/e3bcbc89-2846-48c1-a393-66b635478cce
            
            const patterns = [
                /@?https?:\/\/[^\/]*metricpoints\.com\/api\/csp-reports\/([a-f0-9\-]{36})/,
                /@?[^\/]*metricpoints\.com\/api\/csp-reports\/([a-f0-9\-]{36})/,
                /\/api\/csp-reports\/([a-f0-9\-]{36})/
            ];
            
            for (let pattern of patterns) {
                const match = value.match(pattern);
                if (match && match[1]) {
                    return match[1];
                }
            }
            
            return null;
        }
        
        function showExtractionConfirmation(input) {
            const originalColor = input.style.backgroundColor;
            input.style.backgroundColor = '#d4edda';
            setTimeout(function() {
                input.style.backgroundColor = originalColor;
            }, 1000);
        }
        
        
        // API Key URL stripping
        const apiKeyInput = document.getElementById('metricpoints-csp-api-key');
        if (apiKeyInput) {
            apiKeyInput.addEventListener('paste', function(e) {
                setTimeout(function() {
                    const input = document.getElementById('metricpoints-csp-api-key');
                    let value = input.value;
                    
                    if (value.includes('metricpoints.com') && value.includes('/api/csp-reports/')) {
                        const apiKey = extractApiKey(value);
                        if (apiKey) {
                            input.value = apiKey;
                            showExtractionConfirmation(input);
                        }
                    }
                }, 10);
            });
            
            apiKeyInput.addEventListener('input', function(e) {
                const value = e.target.value;
                
                if (value.includes('metricpoints.com') && value.includes('/api/csp-reports/')) {
                    const apiKey = extractApiKey(value);
                    if (apiKey) {
                        e.target.value = apiKey;
                        showExtractionConfirmation(e.target);
                    }
                }
            });
        }
        
        // Security headers toggle functionality
        // Test Connection Button (for connected users)
        const testConnectionBtn = document.getElementById('test-connection-btn');
        if (testConnectionBtn) {
            testConnectionBtn.addEventListener('click', function() {
                const resultDiv = document.getElementById('test-connection-result');
                const btn = this;
                const originalText = btn.textContent;
                
                btn.disabled = true;
                btn.textContent = '⏳ Testing...';
                
                if (resultDiv) {
                    resultDiv.innerHTML = '<div style="padding: 10px; background: #fff3cd; border-radius: 4px;">Testing connection...</div>';
                }
                
                fetch(metricpointsCspManager.ajaxUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=metricpoints_verify_connection&nonce=' + metricpointsCspManager.nonce
                })
                .then(response => response.json())
                .then(data => {
                    btn.disabled = false;
                    btn.textContent = originalText;
                    
                    if (resultDiv) {
                        if (data.success) {
                            resultDiv.innerHTML = '<div style="padding: 10px; background: #d4edda; border-radius: 4px; color: #155724;"><strong>✅ Connection successful!</strong>';
                            if (data.data.project) {
                                resultDiv.innerHTML += '<br>Project: ' + (data.data.project.name || 'Unknown');
                            }
                            resultDiv.innerHTML += '</div>';
                        } else {
                            resultDiv.innerHTML = '<div style="padding: 10px; background: #f8d7da; border-radius: 4px; color: #721c24;"><strong>❌ ' + (data.data?.message || 'Connection test failed') + '</strong></div>';
                        }
                    }
                })
                .catch(error => {
                    btn.disabled = false;
                    btn.textContent = originalText;
                    
                    if (resultDiv) {
                        resultDiv.innerHTML = '<div style="padding: 10px; background: #f8d7da; border-radius: 4px; color: #721c24;"><strong>❌ Error: ' + error.message + '</strong></div>';
                    }
                });
            });
        }
        
        // Verify Connection Button (for unknown status)
        const verifyConnectionBtn = document.getElementById('verify-connection-btn');
        if (verifyConnectionBtn) {
            verifyConnectionBtn.addEventListener('click', function() {
                const resultDiv = document.getElementById('verify-connection-result');
                const btn = this;
                const originalText = btn.textContent;
                
                btn.disabled = true;
                btn.textContent = '⏳ Verifying...';
                
                if (resultDiv) {
                    resultDiv.innerHTML = '<div style="padding: 10px; background: #fff3cd; border-radius: 4px;">Verifying connection...</div>';
                }
                
                fetch(metricpointsCspManager.ajaxUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=metricpoints_verify_connection&nonce=' + metricpointsCspManager.nonce
                })
                .then(response => response.json())
                .then(data => {
                    btn.disabled = false;
                    btn.textContent = originalText;
                    
                    if (resultDiv) {
                        if (data.success) {
                            resultDiv.innerHTML = '<div style="padding: 10px; background: #d4edda; border-radius: 4px; color: #155724;"><strong>✅ ' + data.data.message + '</strong>';
                            if (data.data.project) {
                                resultDiv.innerHTML += '<br>Project: ' + (data.data.project.name || 'Unknown');
                            }
                            resultDiv.innerHTML += '</div>';
                            
                            // Reload page after 2 seconds to show updated status
                            setTimeout(function() {
                                window.location.reload();
                            }, 2000);
                        } else {
                            resultDiv.innerHTML = '<div style="padding: 10px; background: #f8d7da; border-radius: 4px; color: #721c24;"><strong>❌ ' + (data.data?.message || 'Verification failed') + '</strong></div>';
                        }
                    }
                })
                .catch(error => {
                    btn.disabled = false;
                    btn.textContent = originalText;
                    
                    if (resultDiv) {
                        resultDiv.innerHTML = '<div style="padding: 10px; background: #f8d7da; border-radius: 4px; color: #721c24;"><strong>❌ Error: ' + error.message + '</strong></div>';
                    }
                });
            });
        }
        
        // Registration form toggle
        const registerBtn = document.getElementById('metricpoints-register-btn');
        const registerForm = document.getElementById('metricpoints-registration-form');
        const cancelRegisterBtn = document.getElementById('metricpoints-cancel-register');
        
        if (registerBtn && registerForm) {
            registerBtn.addEventListener('click', function() {
                registerForm.style.display = registerForm.style.display === 'none' ? 'block' : 'none';
            });
        }
        
        if (cancelRegisterBtn && registerForm) {
            cancelRegisterBtn.addEventListener('click', function() {
                registerForm.style.display = 'none';
            });
        }
        
        // Registration form submission
        const registerFormForm = document.getElementById('metricpoints-register-form-form');
        if (registerFormForm) {
            registerFormForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const resultDiv = document.getElementById('metricpoints-register-result');
                const submitBtn = registerFormForm.querySelector('button[type="submit"]');
                const originalText = submitBtn.textContent;
                
                submitBtn.disabled = true;
                submitBtn.textContent = 'Creating Account...';
                
                if (resultDiv) {
                    resultDiv.innerHTML = '<div style="padding: 10px; background: #fff3cd; border-radius: 4px;">Creating your account...</div>';
                }
                
                const formData = new FormData(registerFormForm);
                formData.append('action', 'metricpoints_register');
                formData.append('nonce', metricpointsCspManager.nonce);
                
                fetch(metricpointsCspManager.ajaxUrl, {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    submitBtn.disabled = false;
                    submitBtn.textContent = originalText;
                    
                    if (resultDiv) {
                        if (data.success) {
                            resultDiv.innerHTML = '<div style="padding: 10px; background: #d4edda; border-radius: 4px; color: #155724;"><strong>✅ Account created successfully!</strong><br>Connecting to MetricPoints...</div>';
                            
                            // Auto-fill API key and reload
                            if (data.data && data.data.api_key) {
                                const apiKeyInput = document.getElementById('metricpoints-csp-api-key');
                                if (apiKeyInput) {
                                    apiKeyInput.value = data.data.api_key;
                                }
                                
                                // Reload page after 2 seconds
                                setTimeout(function() {
                                    window.location.reload();
                                }, 2000);
                            }
                        } else {
                            resultDiv.innerHTML = '<div style="padding: 10px; background: #f8d7da; border-radius: 4px; color: #721c24;"><strong>❌ ' + (data.data?.message || 'Registration failed') + '</strong></div>';
                        }
                    }
                })
                .catch(error => {
                    submitBtn.disabled = false;
                    submitBtn.textContent = originalText;
                    
                    if (resultDiv) {
                        resultDiv.innerHTML = '<div style="padding: 10px; background: #f8d7da; border-radius: 4px; color: #721c24;"><strong>❌ Error: ' + error.message + '</strong></div>';
                    }
                });
            });
        }
        
        // Permissions-Policy toggle functionality
        const permissionsPolicyCheckbox = document.querySelector('input[name="metricpoints_csp_manager_options[permissions_policy_enabled]"]');
        if (permissionsPolicyCheckbox) {
            const featuresGrid = document.getElementById('permissions-policy-features-grid');
            if (featuresGrid) {
                // Set initial state
                featuresGrid.style.display = permissionsPolicyCheckbox.checked ? 'grid' : 'none';
                
                // Toggle on change
                permissionsPolicyCheckbox.addEventListener('change', function() {
                    featuresGrid.style.display = this.checked ? 'grid' : 'none';
                });
            }
        }
        
        // Expect-CT toggle functionality
        const expectCtCheckbox = document.querySelector('input[name="metricpoints_csp_manager_options[expect_ct_enabled]"]');
        if (expectCtCheckbox) {
            const expectCtSettings = document.getElementById('expect-ct-settings');
            const expectCtEnforce = document.getElementById('expect-ct-enforce');
            const expectCtReportUri = document.getElementById('expect-ct-report-uri');
            
            [expectCtSettings, expectCtEnforce, expectCtReportUri].forEach(function(element) {
                if (element) {
                    element.style.display = expectCtCheckbox.checked ? 'block' : 'none';
                }
            });
            
            expectCtCheckbox.addEventListener('change', function() {
                [expectCtSettings, expectCtEnforce, expectCtReportUri].forEach(function(element) {
                    if (element) {
                        element.style.display = this.checked ? 'block' : 'none';
                    }
                }.bind(this));
            });
        }
        
        // Clear-Site-Data toggle functionality
        const clearSiteDataCheckbox = document.querySelector('input[name="metricpoints_csp_manager_options[clear_site_data_enabled]"]');
        if (clearSiteDataCheckbox) {
            const clearSiteDataSettings = document.getElementById('clear-site-data-settings');
            if (clearSiteDataSettings) {
                clearSiteDataSettings.style.display = clearSiteDataCheckbox.checked ? 'block' : 'none';
                clearSiteDataCheckbox.addEventListener('change', function() {
                    clearSiteDataSettings.style.display = this.checked ? 'block' : 'none';
                });
            }
        }
        
        // Cross-Origin headers toggle functionality
        const coepCheckbox = document.querySelector('input[name="metricpoints_csp_manager_options[coep_enabled]"]');
        const coopCheckbox = document.querySelector('input[name="metricpoints_csp_manager_options[coop_enabled]"]');
        const corpCheckbox = document.querySelector('input[name="metricpoints_csp_manager_options[corp_enabled]"]');
        
        if (coepCheckbox) {
            const coepSelect = document.getElementById('coep-select');
            const coepDescription = document.getElementById('coep-description');
            if (coepSelect) {
                coepSelect.style.display = coepCheckbox.checked ? 'block' : 'none';
                if (coepDescription) {
                    coepDescription.style.display = coepCheckbox.checked ? 'block' : 'none';
                }
                coepCheckbox.addEventListener('change', function() {
                    const display = this.checked ? 'block' : 'none';
                    coepSelect.style.display = display;
                    if (coepDescription) {
                        coepDescription.style.display = display;
                    }
                });
            }
        }
        
        if (coopCheckbox) {
            const coopSelect = document.getElementById('coop-select');
            const coopDescription = document.getElementById('coop-description');
            if (coopSelect) {
                coopSelect.style.display = coopCheckbox.checked ? 'block' : 'none';
                if (coopDescription) {
                    coopDescription.style.display = coopCheckbox.checked ? 'block' : 'none';
                }
                coopCheckbox.addEventListener('change', function() {
                    const display = this.checked ? 'block' : 'none';
                    coopSelect.style.display = display;
                    if (coopDescription) {
                        coopDescription.style.display = display;
                    }
                });
            }
        }
        
        if (corpCheckbox) {
            const corpSelect = document.getElementById('corp-select');
            const corpDescription = document.getElementById('corp-description');
            if (corpSelect) {
                corpSelect.style.display = corpCheckbox.checked ? 'block' : 'none';
                if (corpDescription) {
                    corpDescription.style.display = corpCheckbox.checked ? 'block' : 'none';
                }
                corpCheckbox.addEventListener('change', function() {
                    const display = this.checked ? 'block' : 'none';
                    corpSelect.style.display = display;
                    if (corpDescription) {
                        corpDescription.style.display = display;
                    }
                });
            }
        }
        
        const securityHeadersToggle = document.getElementById('security-headers-toggle');
        if (securityHeadersToggle) {
            securityHeadersToggle.addEventListener('click', function() {
                const toggle = document.getElementById('security-headers-toggle');
                const settings = document.getElementById('security-headers-settings');
                
                if (settings.style.display === 'none') {
                    settings.style.display = 'block';
                    toggle.textContent = '▲';
                } else {
                    settings.style.display = 'none';
                    toggle.textContent = '▼';
                }
            });
        }
        
        const testCspBtn = document.getElementById('test-csp-btn');
        if (testCspBtn) {
            testCspBtn.addEventListener('click', function() {
                const resultDiv = document.getElementById('test-result');
                if (resultDiv) {
                    resultDiv.innerHTML = '<div style="background: #fff3e0; padding: 15px; border-radius: 5px;"><p>🧪 <strong>Generating Test Violations...</strong></p><p>Creating various types of CSP violations for testing...</p></div>';
                    
                    // Generate multiple types of violations
                    const violations = [];
                    
                    try {
                        // 1. External script violation
                    const script = document.createElement('script');
                        script.src = 'https://cdn.example.com/test-script.js';
                    document.head.appendChild(script);
                        violations.push('External script from cdn.example.com');
                    } catch (e) {
                        violations.push('External script violation (blocked)');
                    }
                    
                    try {
                        // 2. Inline script violation
                        const inlineScript = document.createElement('script');
                        inlineScript.textContent = 'void(0);';
                        document.head.appendChild(inlineScript);
                        violations.push('Inline script execution');
                    } catch (e) {
                        violations.push('Inline script violation (blocked)');
                    }
                    
                    try {
                        // 3. External stylesheet violation
                        const link = document.createElement('link');
                        link.rel = 'stylesheet';
                        link.href = 'https://fonts.googleapis.com/css?family=Test';
                        document.head.appendChild(link);
                        violations.push('External stylesheet from fonts.googleapis.com');
                    } catch (e) {
                        violations.push('External stylesheet violation (blocked)');
                    }
                    
                    try {
                        // 4. Inline style violation (create but don't persist - remove after test)
                        const style = document.createElement('style');
                        style.id = 'metricpoints-test-violation-style';
                        style.textContent = 'body { background: red !important; }';
                        document.head.appendChild(style);
                        violations.push('Inline style execution');
                        // Remove the style after 2 seconds to restore original background
                        setTimeout(() => {
                            const testStyle = document.getElementById('metricpoints-test-violation-style');
                            if (testStyle) {
                                testStyle.remove();
                            }
                        }, 2000);
                    } catch (e) {
                        violations.push('Inline style violation (blocked)');
                    }
                    
                    try {
                        // 5. Image from external domain
                        const img = document.createElement('img');
                        img.src = 'https://via.placeholder.com/150x150.png';
                        img.style.display = 'none';
                        document.body.appendChild(img);
                        violations.push('External image from via.placeholder.com');
                    } catch (e) {
                        violations.push('External image violation (blocked)');
                    }
                    
                    try {
                        // 6. Data URI image
                        const dataImg = document.createElement('img');
                        dataImg.src = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==';
                        dataImg.style.display = 'none';
                        document.body.appendChild(dataImg);
                        violations.push('Data URI image');
                    } catch (e) {
                        violations.push('Data URI image violation (blocked)');
                    }
                    
                    // Show results
                    let html = '<div style="background: #e8f5e8; padding: 15px; border-radius: 5px;">';
                    html += '<p>✅ <strong>Test Violations Generated!</strong></p>';
                    html += '<p>Created ' + violations.length + ' different types of violations:</p>';
                    html += '<ul style="margin: 10px 0; padding-left: 20px;">';
                    
                    violations.forEach(violation => {
                        html += '<li>' + violation + '</li>';
                    });
                    
                    html += '</ul>';
                    html += '<p style="margin-top: 15px; padding: 10px; background: #f0f8ff; border-radius: 4px; border-left: 4px solid #007cba;">';
                    html += '<strong>💡 Next Step:</strong> Click "Analyze Violations & Learn" to see how the AI analyzes these violations and generates recommendations.';
                    html += '</p>';
                    html += '</div>';
                    
                    resultDiv.innerHTML = html;
                    
                    // Update the violation count display
                    setTimeout(() => {
                        updateHealingStats();
                    }, 1000);
                }
            });
        }
        
        const analyzeBtn = document.getElementById('analyze-violations-btn');
        if (analyzeBtn) {
            analyzeBtn.addEventListener('click', function() {
                // Prevent multiple simultaneous clicks
                if (this.disabled) {
                    return;
                }
                
                const resultDiv = document.getElementById('test-result');
                if (resultDiv) {
                    // Disable button and show loading state
                    this.disabled = true;
                    const originalText = this.textContent;
                    this.textContent = '⏳ Analyzing...';
                    
                    resultDiv.innerHTML = '<div style="background: #e3f2fd; padding: 15px; border-radius: 5px;"><p>🧠 <strong>Analyzing Violations...</strong></p><p>Scanning violation data and generating recommendations...</p></div>';
                    
                    // Call AJAX endpoint
                    fetch(metricpointsCspManager.ajaxUrl, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            action: 'metricpoints_analyze_violations',
                            nonce: metricpointsCspManager.nonce
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        // Re-enable button
                        analyzeBtn.disabled = false;
                        analyzeBtn.textContent = originalText;
                        if (data.success) {
                            const results = data.data;
                            let html = '<div style="background: #e8f5e8; padding: 15px; border-radius: 5px;">';
                            html += '<p>✅ <strong>Analysis Complete!</strong></p>';
                            html += '<p><strong>' + results.message + '</strong></p>';
                            
                            if (results.violations_found === 0) {
                                html += '<div style="background: #fff3e0; padding: 10px; border-radius: 4px; margin: 10px 0; border-left: 4px solid #ff9800;">';
                                html += '<p><strong>⚠️ No Violations Found</strong></p>';
                                html += '<p>To test the analysis feature:</p>';
                                html += '<ul style="margin: 5px 0; padding-left: 20px;">';
                                html += '<li>Click "Generate CSP Violation" to create test violations</li>';
                                html += '<li>Make sure your CSP policy is active and blocking resources</li>';
                                html += '<li>Visit your site and trigger some violations</li>';
                                html += '</ul>';
                                html += '</div>';
                            } else {
                                // Show violation statistics
                                html += '<div style="background: #f0f8ff; padding: 10px; border-radius: 4px; margin: 10px 0; border-left: 4px solid #007cba;">';
                                html += '<p><strong>📊 Violation Statistics:</strong></p>';
                                html += '<ul style="margin: 5px 0; padding-left: 20px;">';
                                
                                for (const [directive, count] of Object.entries(results.directive_violations || {})) {
                                    html += '<li><strong>' + directive + ':</strong> ' + count + ' violations</li>';
                                }
                                
                                html += '</ul>';
                                html += '<p><strong>Confidence Score:</strong> ' + Math.round(results.confidence_score * 100) + '%</p>';
                                html += '</div>';
                                
                                // Show recommendations
                                if (results.recommendations && results.recommendations.length > 0) {
                                    html += '<div style="background: #f8f9fa; padding: 10px; border-radius: 4px; margin: 10px 0; border-left: 4px solid #28a745;">';
                                    html += '<p><strong>💡 Recommendations:</strong></p>';
                                    html += '<ul style="margin: 5px 0; padding-left: 20px;">';
                                    
                                    results.recommendations.forEach(rec => {
                                        html += '<li>' + rec + '</li>';
                                    });
                                    
                                    html += '</ul>';
                                    html += '</div>';
                                }
                                
                                // Show blocked URIs
                                if (results.blocked_uris && results.blocked_uris.length > 0) {
                                    html += '<div style="background: #fff3e0; padding: 10px; border-radius: 4px; margin: 10px 0; border-left: 4px solid #ff9800;">';
                                    html += '<p><strong>🚫 Blocked URIs (first 10):</strong></p>';
                                    html += '<ul style="margin: 5px 0; padding-left: 20px; font-family: monospace; font-size: 12px;">';
                                    
                                    results.blocked_uris.forEach(uri => {
                                        html += '<li>' + uri + '</li>';
                                    });
                                    
                                    html += '</ul>';
                                    html += '</div>';
                                }
                                
                                // Show external domains
                                if (results.external_domains && results.external_domains.length > 0) {
                                    html += '<div style="background: #e8f5e8; padding: 10px; border-radius: 4px; margin: 10px 0; border-left: 4px solid #4caf50;">';
                                    html += '<p><strong>🌐 External Domains Found:</strong></p>';
                                    html += '<p style="font-family: monospace; font-size: 12px; background: #f5f5f5; padding: 5px; border-radius: 3px;">';
                                    html += results.external_domains.join(', ');
                                    html += '</p>';
                                    html += '</div>';
                                }
                            }
                            
                            html += '</div>';
                            resultDiv.innerHTML = html;
                        updateHealingStats();
                        } else {
                            resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Analysis Failed</strong></p><p>' + (data.data || 'Unknown error occurred') + '</p></div>';
                            // Re-enable button on error
                            analyzeBtn.disabled = false;
                            analyzeBtn.textContent = originalText;
                        }
                    })
                    .catch(error => {
                        resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Analysis Failed</strong></p><p>Error: ' + error.message + '</p></div>';
                        // Re-enable button on error
                        analyzeBtn.disabled = false;
                        analyzeBtn.textContent = originalText;
                    });
                }
            });
        }
        
        const forceUpdateBtn = document.getElementById('force-update-btn');
        if (forceUpdateBtn) {
            forceUpdateBtn.addEventListener('click', function() {
                const resultDiv = document.getElementById('test-result');
                if (resultDiv) {
                    resultDiv.innerHTML = '<div style="background: #fff3e0; padding: 15px; border-radius: 5px;"><p>⚡ <strong>Force Updating Policy...</strong></p><p>Applying latest learned patterns and optimizations...</p></div>';
                    
                    // Call AJAX endpoint
                    fetch(metricpointsCspManager.ajaxUrl, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            action: 'metricpoints_force_update',
                            nonce: metricpointsCspManager.nonce
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            resultDiv.innerHTML = '<div style="background: #e8f5e8; padding: 15px; border-radius: 5px;"><p>✅ <strong>Policy Force Updated!</strong></p><p>' + data.data.message + '</p><p><strong>Updated Policy:</strong><br><code style="background: #f5f5f5; padding: 5px; border-radius: 3px; display: block; margin-top: 10px;">' + data.data.updated_policy + '</code></p></div>';
                        updateHealingStats();
                            location.reload(); // Refresh to show updated policy
                        } else {
                            resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Force Update Failed</strong></p><p>' + (data.data || 'Unknown error occurred') + '</p></div>';
                        }
                    })
                    .catch(error => {
                        resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Force Update Failed</strong></p><p>Error: ' + error.message + '</p></div>';
                    });
                }
            });
        }
        
        const resetBtn = document.getElementById('reset-learning-btn');
        if (resetBtn) {
            resetBtn.addEventListener('click', function() {
                if (confirm('Are you sure you want to reset all learning data? This will clear the AI\'s knowledge base and start fresh.')) {
                    const resultDiv = document.getElementById('test-result');
                    if (resultDiv) {
                        resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>🔄 <strong>Resetting Learning Data...</strong></p><p>Clearing AI knowledge base and starting fresh...</p></div>';
                        
                        // Call AJAX endpoint
                        fetch(metricpointsCspManager.ajaxUrl, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/x-www-form-urlencoded',
                            },
                            body: new URLSearchParams({
                                action: 'metricpoints_reset_learning',
                                nonce: metricpointsCspManager.nonce
                            })
                        })
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                resultDiv.innerHTML = '<div style="background: #e8f5e8; padding: 15px; border-radius: 5px;"><p>✅ <strong>Learning Data Reset!</strong></p><p>' + data.data.message + '</p></div>';
                            updateHealingStats();
                                location.reload(); // Refresh to show reset stats
                            } else {
                                resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Reset Failed</strong></p><p>' + (data.data || 'Unknown error occurred') + '</p></div>';
                            }
                        })
                        .catch(error => {
                            resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Reset Failed</strong></p><p>Error: ' + error.message + '</p></div>';
                        });
                    }
                }
            });
        }
        
        const rollbackBtn = document.getElementById('rollback-btn');
        if (rollbackBtn) {
            rollbackBtn.addEventListener('click', function() {
                const resultDiv = document.getElementById('test-result');
                if (resultDiv) {
                    resultDiv.innerHTML = '<div style="background: #fff3e0; padding: 15px; border-radius: 5px;"><p>↩️ <strong>Loading Rollback Options...</strong></p><p>Fetching available policy backups...</p></div>';
                    
                    // Fetch available backups
                    fetch(metricpointsCspManager.ajaxUrl, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            action: 'metricpoints_get_backups',
                            nonce: metricpointsCspManager.nonce
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success && data.data.backups.length > 0) {
                            let optionsHtml = '<div style="max-height: 300px; overflow-y: auto; border: 1px solid #ddd; border-radius: 4px; padding: 10px; margin: 10px 0;">';
                            
                            data.data.backups.forEach((backup, index) => {
                                const timestamp = new Date(backup.timestamp).toLocaleString();
                                const policyPreview = backup.old_policy.length > 100 ? 
                                    backup.old_policy.substring(0, 100) + '...' : 
                                    backup.old_policy;
                                
                                optionsHtml += `
                                    <div style="border: 1px solid #eee; padding: 10px; margin: 5px 0; border-radius: 4px; cursor: pointer; background: #f9f9f9;" 
                                         onclick="rollbackToBackup('${backup.backup_id}')">
                                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 5px;">
                                            <strong>Backup #${index + 1}</strong>
                                            <span style="color: #666; font-size: 12px;">${timestamp}</span>
                                        </div>
                                        <div style="font-family: monospace; font-size: 12px; color: #666; background: #f5f5f5; padding: 5px; border-radius: 3px;">
                                            ${policyPreview}
                                        </div>
                                    </div>
                                `;
                            });
                            
                            optionsHtml += '</div>';
                            optionsHtml += '<p style="color: #666; font-size: 12px; margin-top: 10px;">Click on a backup to restore it.</p>';
                            
                            resultDiv.innerHTML = '<div style="background: #fff3e0; padding: 15px; border-radius: 5px;"><p>↩️ <strong>Rollback Options</strong></p><p>Select a backup to rollback to:</p>' + optionsHtml + '</div>';
                        } else {
                            resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>No Backups Available</strong></p><p>No policy backups found. Create some policy updates first.</p></div>';
                        }
                    })
                    .catch(error => {
                        resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Failed to Load Backups</strong></p><p>Error: ' + error.message + '</p></div>';
                    });
                }
            });
        }
        
        // Rollback function
        window.rollbackToBackup = function(backupId) {
            const resultDiv = document.getElementById('test-result');
            if (resultDiv) {
                resultDiv.innerHTML = '<div style="background: #fff3e0; padding: 15px; border-radius: 5px;"><p>↩️ <strong>Rolling Back Policy...</strong></p><p>Restoring selected backup...</p></div>';
                
                fetch(metricpointsCspManager.ajaxUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'metricpoints_rollback_policy',
                        backup_id: backupId,
                        nonce: metricpointsCspManager.nonce
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        resultDiv.innerHTML = '<div style="background: #e8f5e8; padding: 15px; border-radius: 5px;"><p>✅ <strong>Policy Rolled Back!</strong></p><p>' + data.data.message + '</p><p><strong>Restored Policy:</strong><br><code style="background: #f5f5f5; padding: 5px; border-radius: 3px; display: block; margin-top: 10px;">' + data.data.restored_policy + '</code></p></div>';
                        updateHealingStats();
                        location.reload(); // Refresh to show restored policy
                    } else {
                        resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Rollback Failed</strong></p><p>' + (data.data || 'Unknown error occurred') + '</p></div>';
                    }
                })
                .catch(error => {
                    resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Rollback Failed</strong></p><p>Error: ' + error.message + '</p></div>';
                });
            }
        }
        
        function updateHealingStats() {
            // This would normally fetch updated stats via AJAX
            const violationsCount = document.getElementById('violations-count');
            const updatesCount = document.getElementById('updates-count');
            const lastUpdate = document.getElementById('last-update');
            
            if (violationsCount) {
                const current = parseInt(violationsCount.textContent) || 0;
                violationsCount.textContent = current + Math.floor(Math.random() * 3) + 1;
            }
            
            if (updatesCount) {
                const current = parseInt(updatesCount.textContent) || 0;
                updatesCount.textContent = current + 1;
            }
            
            if (lastUpdate) {
                lastUpdate.textContent = new Date().toLocaleString();
            }
        }
        
        // Backup management functions
        function previewBackup(index) {
            const previewDiv = document.getElementById('backup-preview');
            if (previewDiv) {
                previewDiv.style.display = 'block';
                previewDiv.innerHTML = '<div style="background: #e3f2fd; padding: 15px; border-radius: 5px; border-left: 4px solid #2196f3;"><p>👁️ <strong>Loading Backup Preview #' + (index + 1) + '...</strong></p></div>';
                
                // Fetch the actual backup data via AJAX
                fetch(metricpointsCspManager.ajaxUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=metricpoints_get_backups&nonce=' + metricpointsCspManager.nonce
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.data.backups && data.data.backups[index]) {
                        const backup = data.data.backups[index];
                        const timestamp = new Date(backup.timestamp).toLocaleString();
                        
                        previewDiv.innerHTML = `
                            <div style="background: #f0f8ff; padding: 15px; border-radius: 5px; border-left: 4px solid #007cba;">
                                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                                    <h4 style="margin: 0; color: #007cba;">👁️ Backup Preview #${index + 1}</h4>
                                    <button onclick="document.getElementById('backup-preview').style.display='none'" style="background: #dc3545; color: white; border: none; padding: 5px 10px; border-radius: 3px; cursor: pointer; font-size: 12px;">✕ Close</button>
                                </div>
                                <div style="margin-bottom: 15px; background: #f8f9fa; padding: 10px; border-radius: 4px;">
                                    <strong>📅 Date:</strong> ${timestamp}<br>
                                    <strong>📝 Reason:</strong> ${backup.reason || 'Unknown'}<br>
                                    <strong>⚡ Action:</strong> ${backup.action || 'Unknown'}
                                </div>
                                <div style="background: #ffffff; padding: 15px; border-radius: 4px; border: 1px solid #dee2e6; margin-bottom: 15px;">
                                    <strong>📋 Policy Content:</strong>
                                    <pre style="margin: 10px 0 0 0; font-family: 'Courier New', monospace; font-size: 13px; color: #333; white-space: pre-wrap; word-break: break-all; background: #f8f9fa; padding: 10px; border-radius: 3px; border: 1px solid #e9ecef;">${backup.old_policy || 'No policy content'}</pre>
                                </div>
                                ${backup.changes && backup.changes.length > 0 ? `
                                    <div style="background: #e8f5e8; padding: 10px; border-radius: 4px; border-left: 3px solid #28a745;">
                                        <strong>🔄 Changes Made:</strong>
                                        <ul style="margin: 8px 0 0 20px; font-size: 13px; color: #155724;">
                                            ${backup.changes.map(change => 
                                                `<li style="margin-bottom: 5px;"><strong>${change.directive || 'unknown'}:</strong> <code style="background: #d4edda; padding: 2px 4px; border-radius: 2px;">${change.old_value || 'not set'}</code> → <code style="background: #d4edda; padding: 2px 4px; border-radius: 2px;">${change.new_value || 'not set'}</code></li>`
                                            ).join('')}
                                        </ul>
                                    </div>
                                ` : ''}
                            </div>
                        `;
                    } else {
                        previewDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px; border-left: 4px solid #dc3545;"><p>❌ <strong>Error</strong></p><p>Could not load backup #' + index + '. Please try again.</p></div>';
                    }
                })
                .catch(error => {
                    console.error('Preview error:', error);
                    previewDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px; border-left: 4px solid #dc3545;"><p>❌ <strong>Error</strong></p><p>Failed to load backup preview: ' + error.message + '</p></div>';
                });
            }
        }
        
        function restoreBackup(index) {
            if (confirm('Are you sure you want to restore backup #' + index + '? This will replace your current CSP policy.')) {
                const resultDiv = document.getElementById('test-result');
                if (resultDiv) {
                    resultDiv.innerHTML = '<div style="background: #fff3e0; padding: 15px; border-radius: 5px;"><p>↩️ <strong>Restoring Backup...</strong></p><p>Restoring backup #' + index + '...</p></div>';
                    
                    // First get the backup data to find the backup_id
                    fetch(metricpointsCspManager.ajaxUrl, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'action=metricpoints_get_backups&nonce=' + metricpointsCspManager.nonce
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success && data.data.backups && data.data.backups[index]) {
                            const backup = data.data.backups[index];
                            
                            // Now restore the backup
                            return fetch(metricpointsCspManager.ajaxUrl, {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/x-www-form-urlencoded',
                                },
                                body: 'action=metricpoints_rollback_policy&backup_id=' + encodeURIComponent(backup.backup_id) + '&nonce=' + metricpointsCspManager.nonce
                            });
                        } else {
                            throw new Error('Backup not found');
                        }
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            resultDiv.innerHTML = '<div style="background: #d4edda; padding: 15px; border-radius: 5px;"><p>✅ <strong>Backup Restored Successfully</strong></p><p>Policy has been restored from backup #' + index + '.</p><p><strong>Restored Policy:</strong><br><code style="background: #f8f9fa; padding: 5px; border-radius: 3px; font-size: 12px;">' + (data.data.restored_policy || 'Unknown') + '</code></p></div>';
                            
                            // Refresh the page after a short delay
                            setTimeout(() => {
                                window.location.reload();
                            }, 2000);
                        } else {
                            resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Restore Failed</strong></p><p>Error: ' + (data.data || 'Unknown error') + '</p></div>';
                        }
                    })
                    .catch(error => {
                        console.error('Restore error:', error);
                        resultDiv.innerHTML = '<div style="background: #ffebee; padding: 15px; border-radius: 5px;"><p>❌ <strong>Restore Failed</strong></p><p>Error: ' + error.message + '</p></div>';
                    });
                }
            }
        }
        
        
        // CSP Policy validation message
        const cspTextarea = document.getElementById('csp-policy-textarea');
        const validationMessage = document.getElementById('csp-validation-message');
        let originalValue = cspTextarea ? cspTextarea.value : '';
        
        if (cspTextarea && validationMessage) {
            cspTextarea.addEventListener('input', function() {
                if (this.value !== originalValue) {
                    validationMessage.style.display = 'block';
                } else {
                    validationMessage.style.display = 'none';
                }
            });
        }
        
        // Webhook URL copy function
        window.copyWebhookUrl = function() {
            const webhookUrl = '<?php echo $this->get_self_healing_webhook_url(); ?>';
            navigator.clipboard.writeText(webhookUrl).then(function() {
                alert('Webhook URL copied to clipboard!');
            }).catch(function(err) {
                console.error('Could not copy webhook URL: ', err);
                alert('Could not copy to clipboard. Please copy manually.');
            });
        }
        
        // Test webhook function - tests MetricPoints webhook endpoint via server-side AJAX
        window.testWebhook = function() {
            const apiKey = '<?php echo esc_js($this->options['api_key'] ?? ''); ?>';
            const metricpointsWebhookUrl = '<?php echo $this->get_self_healing_webhook_url(); ?>';
            
            if (!apiKey) {
                alert('Please configure an API key first.');
                return;
            }
            
            if (!confirm('This will send a test webhook to MetricPoints API endpoint:\n\n' + metricpointsWebhookUrl + '\n\nThis simulates a real webhook that would trigger CSP policy analysis. Continue?')) {
                return;
            }
            
            // Show loading state
            const testButton = event.target;
            const originalText = testButton.textContent;
            testButton.textContent = '🔄 Testing...';
            testButton.disabled = true;
            
            // Use server-side AJAX to avoid CORS issues
            fetch(metricpointsCspManager.ajaxUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=metricpoints_test_webhook&nonce=' + metricpointsCspManager.nonce
            })
            .then(response => response.json())
            .then(data => {
                // Reset button state
                testButton.textContent = originalText;
                testButton.disabled = false;
                
                if (data.success) {
                    let message = '✅ Webhook test successful!\n\n';
                    message += 'Response Code: ' + data.data.response_code + '\n';
                    message += 'Webhook URL: ' + data.data.webhook_url + '\n\n';
                    message += 'Response: ' + JSON.stringify(data.data.response, null, 2);
                    
                    alert(message);
                    
                    // Optionally refresh to show any updates
                    if (confirm('Refresh the page to see any policy updates?')) {
                        location.reload();
                    }
                } else {
                    let message = '❌ Webhook test failed!\n\n';
                    if (data.data && data.data.response_code) {
                        message += 'Response Code: ' + data.data.response_code + '\n';
                    }
                    if (data.data && data.data.webhook_url) {
                        message += 'Webhook URL: ' + data.data.webhook_url + '\n\n';
                    }
                    message += 'Error: ' + (data.data.message || data.message || 'Unknown error') + '\n\n';
                    if (data.data && data.data.response) {
                        message += 'Response: ' + JSON.stringify(data.data.response, null, 2);
                    }
                    if (data.data && data.data.debug) {
                        message += '\n\nDebug Info:\n';
                        message += 'Response Body: ' + data.data.debug.response_body + '\n';
                        message += 'JSON Decode Error: ' + data.data.debug.json_decode_error;
                    }
                    
                    alert(message);
                }
            })
            .catch(error => {
                // Reset button state
                testButton.textContent = originalText;
                testButton.disabled = false;
                
                console.error('Webhook test error:', error);
                alert('❌ Webhook test failed: ' + error.message + '\n\nThis could be due to:\n• Network connectivity issues\n• WordPress AJAX error\n• Server configuration issues');
            });
        }
        
        // Activate skipped update
        function activateSkippedUpdate(updateIndex) {
            if (!confirm('Are you sure you want to activate this skipped policy update? This will change your current CSP policy.')) {
                return;
            }
            
            fetch(metricpointsCspManager.ajaxUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=metricpoints_activate_skipped_update&update_index=' + updateIndex + '&nonce=' + metricpointsCspManager.nonce
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('✅ Policy update activated successfully!');
                    location.reload(); // Refresh to show updated status
                } else {
                    alert('❌ Failed to activate update: ' + (data.data || 'Unknown error'));
                }
            })
            .catch(error => {
                console.error('Activation error:', error);
                alert('❌ Error activating update: ' + error.message);
            });
        }
        
        // Backup pagination functionality
        document.addEventListener('DOMContentLoaded', function() {
            const showMoreBtn = document.getElementById('show-more-backups');
            const showRecentBtn = document.getElementById('show-recent-backups');
            const policyBackupsDiv = document.getElementById('policy-backups');
            
            if (showMoreBtn && showRecentBtn && policyBackupsDiv) {
                showMoreBtn.addEventListener('click', function() {
                    // Fetch all backups
                    fetch(metricpointsCspManager.ajaxUrl, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'action=metricpoints_get_backups&show_all=1&nonce=' + metricpointsCspManager.nonce
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success && data.data.backups) {
                            // Update the backup display with all backups
                            policyBackupsDiv.innerHTML = data.data.html;
                            
                            // Toggle buttons
                            showMoreBtn.style.display = 'none';
                            showRecentBtn.style.display = 'inline-block';
                            
                            // Update the status message
                            const statusMsg = policyBackupsDiv.querySelector('p[style*="color: #666"]');
                            if (statusMsg) {
                                statusMsg.textContent = 'Showing all ' + data.data.backups.length + ' backups.';
                            }
                        }
                    })
                    .catch(error => {
                        console.error('Error loading all backups:', error);
                        alert('Error loading all backups. Please try again.');
                    });
                });
                
                showRecentBtn.addEventListener('click', function() {
                    // Reload the page to show only recent backups
                    window.location.reload();
                });
            }
        });
        </script>
        <?php
    }
    
    public function add_csp_header() {
        if (empty($this->options['enabled'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP: Headers not added - plugin disabled');
            }
            return;
        }
        
        // Check if we're on an admin page and admin headers are disabled
        $is_admin_page = is_admin() && !wp_doing_ajax();
        if ($is_admin_page && empty($this->options['admin_headers'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP: Headers not added - admin headers disabled');
            }
            return;
        }
        
        // Only add headers if we haven't already sent any output
        if (headers_sent()) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP: Headers already sent, cannot add CSP headers');
            }
            return;
        }
        
        $csp_policy = $this->get_current_csp_policy();
        
        if (!empty($csp_policy)) {
            // Validate the policy before sending headers
            $validation = $this->validate_csp_policy($csp_policy);
            
            if (!$validation['valid']) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('MetricPoints CSP: Policy validation failed - not sending headers');
                    error_log('MetricPoints CSP: Validation errors: ' . implode(', ', $validation['errors']));
                }
                return;
            }
            
            $csp_mode = isset($this->options['csp_mode']) ? $this->options['csp_mode'] : 'enforcing';
            
            if ($csp_mode === 'report-only') {
                header("Content-Security-Policy-Report-Only: " . $csp_policy);
            } else {
                header("Content-Security-Policy: " . $csp_policy);
            }
        }
        
        // Add other security headers
        $this->add_security_headers();
    }
    
    private function add_security_headers() {
        // HSTS
        if (!empty($this->options['hsts_enabled'])) {
            $max_age = isset($this->options['hsts_max_age']) ? $this->options['hsts_max_age'] : 31536000;
            $include_subdomains = !empty($this->options['hsts_include_subdomains']) ? '; includeSubDomains' : '';
            $preload = !empty($this->options['hsts_preload']) ? '; preload' : '';
            
            header("Strict-Transport-Security: max-age={$max_age}{$include_subdomains}{$preload}");
        }
        
        // X-Frame-Options
        if (!empty($this->options['x_frame_options'])) {
            $value = $this->options['x_frame_options'];
            if ($value === 'ALLOW-FROM') {
                // Note: ALLOW-FROM requires a URI, but we'll use SAMEORIGIN as fallback
                $value = 'SAMEORIGIN';
            }
            header("X-Frame-Options: {$value}");
        }
        
        // X-Content-Type-Options
        if (!empty($this->options['x_content_type_options'])) {
            header("X-Content-Type-Options: nosniff");
        }
        
        // Referrer-Policy
        if (!empty($this->options['referrer_policy'])) {
            header("Referrer-Policy: " . $this->options['referrer_policy']);
        }
        
        // X-XSS-Protection
        if (!empty($this->options['x_xss_protection'])) {
            header("X-XSS-Protection: " . $this->options['x_xss_protection']);
        }
        
        // Permissions-Policy
        if (!empty($this->options['permissions_policy_enabled'])) {
            $features = isset($this->options['permissions_policy_features']) ? $this->options['permissions_policy_features'] : array();
            if (!empty($features)) {
                $policy_parts = array();
                foreach ($features as $feature => $value) {
                    if ($value === 'none') {
                        $policy_parts[] = $feature . '=()';
                    } elseif ($value === 'self') {
                        $policy_parts[] = $feature . '=(self)';
                    } elseif ($value === '*') {
                        $policy_parts[] = $feature . '=*';
                    }
                }
                if (!empty($policy_parts)) {
                    header("Permissions-Policy: " . implode(', ', $policy_parts));
                }
            }
        }
        
        // Expect-CT
        if (!empty($this->options['expect_ct_enabled'])) {
            $max_age = isset($this->options['expect_ct_max_age']) ? $this->options['expect_ct_max_age'] : 86400;
            $expect_ct_value = "max-age={$max_age}";
            if (!empty($this->options['expect_ct_enforce'])) {
                $expect_ct_value .= ', enforce';
            }
            if (!empty($this->options['expect_ct_report_uri'])) {
                $report_uri = esc_url_raw($this->options['expect_ct_report_uri']);
                $expect_ct_value .= ', report-uri="' . $report_uri . '"';
            }
            header("Expect-CT: {$expect_ct_value}");
        }
        
        // Clear-Site-Data
        if (!empty($this->options['clear_site_data_enabled'])) {
            $directives = isset($this->options['clear_site_data_directives']) ? $this->options['clear_site_data_directives'] : array();
            if (!empty($directives)) {
                $clear_directives = array();
                foreach ($directives as $directive => $enabled) {
                    if ($enabled) {
                        $clear_directives[] = '"' . $directive . '"';
                    }
                }
                if (!empty($clear_directives)) {
                    header("Clear-Site-Data: " . implode(', ', $clear_directives));
                }
            }
        }
        
        // Cross-Origin-Embedder-Policy (COEP)
        if (!empty($this->options['coep_enabled'])) {
            $coep_value = isset($this->options['coep_value']) ? $this->options['coep_value'] : 'require-corp';
            header("Cross-Origin-Embedder-Policy: {$coep_value}");
        }
        
        // Cross-Origin-Opener-Policy (COOP)
        if (!empty($this->options['coop_enabled'])) {
            $coop_value = isset($this->options['coop_value']) ? $this->options['coop_value'] : 'same-origin';
            header("Cross-Origin-Opener-Policy: {$coop_value}");
        }
        
        // Cross-Origin-Resource-Policy (CORP)
        if (!empty($this->options['corp_enabled'])) {
            $corp_value = isset($this->options['corp_value']) ? $this->options['corp_value'] : 'same-origin';
            header("Cross-Origin-Resource-Policy: {$corp_value}");
        }
        
        // Custom Headers
        if (!empty($this->options['custom_headers'])) {
            $this->add_custom_headers();
        }
    }
    
    private function add_custom_headers() {
        $custom_headers = $this->options['custom_headers'];
        
        // Split by lines and process each header
        $lines = explode("\n", $custom_headers);
        
        foreach ($lines as $line) {
            $line = trim($line);
            
            // Skip empty lines
            if (empty($line)) {
                continue;
            }
            
            // Skip comment lines (starting with #)
            if (strpos($line, '#') === 0) {
                continue;
            }
            
            // Check if line contains a colon (header format)
            if (strpos($line, ':') !== false) {
                // Split header name and value
                $parts = explode(':', $line, 2);
                $header_name = trim($parts[0]);
                $header_value = trim($parts[1]);
                
                // Validate header name (basic validation)
                if (!empty($header_name) && !empty($header_value)) {
                    // Send the custom header
                    header("{$header_name}: {$header_value}");
                }
            }
        }
    }
    
    private function validate_csp_policy($policy) {
        $errors = array();
        $warnings = array();
        
        // Remove comments and clean up
        $clean_policy = preg_replace('/#.*$/m', '', $policy);
        $clean_policy = preg_replace('/\s+/', ' ', trim($clean_policy));
        
        // Split into directives
        $directives = explode(';', $clean_policy);
        
        // Valid directive names
        $valid_directives = array(
            'default-src', 'script-src', 'style-src', 'img-src', 'font-src', 'connect-src',
            'media-src', 'object-src', 'child-src', 'frame-src', 'worker-src', 'manifest-src',
            'base-uri', 'form-action', 'frame-ancestors', 'navigate-to', 'report-uri', 'report-to',
            'upgrade-insecure-requests', 'block-all-mixed-content', 'require-sri-for',
            'trusted-types', 'require-trusted-types-for'
        );
        
        // Common source values
        $valid_sources = array(
            'self', 'unsafe-inline', 'unsafe-eval', 'unsafe-hashes', 'unsafe-allow-redirects',
            'none', 'strict-dynamic', 'report-sample', 'data', 'blob', 'filesystem',
            'https:', 'http:', 'https://', 'http://'
        );
        
        foreach ($directives as $directive) {
            $directive = trim($directive);
            if (empty($directive)) continue;
            
            // Check if directive has a value
            if (strpos($directive, ' ') === false) {
                $errors[] = "Directive '{$directive}' has no value";
                continue;
            }
            
            // Split directive name and values
            $parts = explode(' ', $directive, 2);
            $directive_name = trim($parts[0]);
            $directive_values = trim($parts[1]);
            
            // Check if directive name is valid
            if (!in_array($directive_name, $valid_directives)) {
                $errors[] = "Unknown directive '{$directive_name}'. Valid directives: " . implode(', ', array_slice($valid_directives, 0, 10)) . "...";
            }
            
            // Check for common typos in source values
            $sources = explode(' ', $directive_values);
            foreach ($sources as $source) {
                $source = trim($source, "'\"");
                
                // Check for common typos
                $typos = array(
                    'selfie' => 'self',
                    'unsafe-hash' => 'unsafe-hashes',
                    'unsafe-redirect' => 'unsafe-allow-redirects',
                    'unsafe-allow-redirect' => 'unsafe-allow-redirects'
                );
                
                if (isset($typos[$source])) {
                    $errors[] = "Typo detected: '{$source}' should be '{$typos[$source]}'";
                }
                
                // Check for URLs that don't start with http/https/data/
                if (!preg_match('/^(https?:\/\/|https?:|data:|blob:|filesystem:|\'self\'|\'unsafe-|\'strict-|\'report-|\'none\'|[\'"]*[\w\-]+[\'"]*$)/', $source)) {
                    if (!in_array($source, $valid_sources) && !preg_match('/^[\'"]*[\w\-]+[\'"]*$/', $source)) {
                        $warnings[] = "Suspicious source value: '{$source}' - make sure this is correct";
                    }
                }
            }
        }
        
        return array(
            'valid' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings
        );
    }
    
    public function register_webhook_endpoint() {
        register_rest_route('metricpoints-csp/v1', '/webhook', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_webhook'),
            'permission_callback' => array($this, 'webhook_permission_check'),
            'args' => array(
                'webhook_type' => array(
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ),
                'timestamp' => array(
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ),
                'project' => array(
                    'required' => true,
                    'type' => 'object',
                ),
                'policy_update' => array(
                    'required' => true,
                    'type' => 'object',
                ),
            ),
        ));
        
        // Add status endpoint for testing
        register_rest_route('metricpoints-csp/v1', '/status', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_status_check'),
            'permission_callback' => '__return_true', // Public endpoint for testing
        ));
    }
    
    /**
     * Handle status check endpoint
     */
    public function handle_status_check($request) {
        $api_key = isset($this->options['api_key']) ? $this->options['api_key'] : '';
        $enabled = isset($this->options['enabled']) ? $this->options['enabled'] : 0;
        $self_healing = isset($this->options['self_healing']) ? $this->options['self_healing'] : 0;
        $csp_policy = isset($this->options['csp_policy']) ? $this->options['csp_policy'] : '';
        
        return new WP_REST_Response(array(
            'plugin_active' => true,
            'plugin_version' => METRICPOINTS_CSP_MANAGER_VERSION,
            'configured' => !empty($api_key) && $enabled,
            'self_healing_enabled' => (bool)$self_healing,
            'has_api_key' => !empty($api_key),
            'csp_policy_set' => !empty($csp_policy),
            'webhook_endpoint' => rest_url('metricpoints-csp/v1/webhook'),
            'status_endpoint' => rest_url('metricpoints-csp/v1/status'),
            'timestamp' => current_time('mysql'),
            'wordpress_version' => get_bloginfo('version'),
            'site_url' => get_site_url()
        ), 200);
    }
    
    public function webhook_permission_check($request) {
        // Check if the request has the correct API key
        $api_key = $request->get_header('X-MetricPoints-API-Key');
        $stored_api_key = isset($this->options['api_key']) ? $this->options['api_key'] : '';
        
        if (empty($api_key) || $api_key !== $stored_api_key) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints WordPress Webhook: Invalid API key');
            }
            return new WP_Error('invalid_api_key', 'Invalid API key', array('status' => 401));
        }
        
        // Check if self-healing is enabled
        if (empty($this->options['self_healing'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints WordPress Webhook: Self-healing not enabled');
            }
            return new WP_Error('self_healing_disabled', 'Self-healing is not enabled', array('status' => 403));
        }
        
        return true;
    }
    
    public function handle_webhook($request) {
        $data = $request->get_json_params();
        
        // Validate webhook type
        if (!isset($data['webhook_type']) || $data['webhook_type'] !== 'wordpress_csp_update') {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints WordPress Webhook: Invalid webhook type');
            }
            return new WP_Error('invalid_webhook_type', 'Invalid webhook type', array('status' => 400));
        }
        
        // Extract policy update data
        $policy_update = $data['policy_update'] ?? array();
        $new_policy = $policy_update['new_policy'] ?? '';
        $action = $policy_update['action'] ?? '';
        $reason = $policy_update['reason'] ?? '';
        $confidence_score = $policy_update['confidence_score'] ?? 0;
        
        if (empty($new_policy)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints WordPress Webhook: No new policy provided');
            }
            return new WP_Error('no_policy', 'No new policy provided', array('status' => 400));
        }
        
        // Validate the new policy
        $validation = $this->validate_csp_policy($new_policy);
        if (!$validation['valid']) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints WordPress Webhook: Invalid policy - ' . implode(', ', $validation['errors']));
            }
            return new WP_Error('invalid_policy', 'Invalid CSP policy: ' . implode(', ', $validation['errors']), array('status' => 400));
        }
        
        // Check confidence threshold
        $confidence_threshold = isset($this->options['confidence_threshold']) ? $this->options['confidence_threshold'] : 80;
        $confidence_percentage = round($confidence_score * 100);
        
        if ($confidence_percentage < $confidence_threshold) {
            // Log as skipped update
            $this->log_policy_update($policy_update, $data, 'skipped');
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints WordPress Webhook: Update skipped due to low confidence - ' . $confidence_percentage . '% < ' . $confidence_threshold . '%');
            }
            return new WP_REST_Response(array(
                'success' => true,
                'message' => 'Policy update skipped due to low confidence',
                'confidence_score' => $confidence_score,
                'confidence_threshold' => $confidence_threshold / 100,
                'skipped' => true,
                'timestamp' => current_time('mysql')
            ), 200);
        }
        
        // Update the CSP policy
        $this->options['csp_policy'] = $new_policy;
        update_option('metricpoints_csp_manager_options', $this->options);
        
        // Log the policy update
        $this->log_policy_update($policy_update, $data, 'applied');
        
        // Send success response
        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'CSP policy updated successfully',
            'updated_policy' => $new_policy,
            'confidence_score' => $confidence_score,
            'timestamp' => current_time('mysql')
        ), 200);
    }
    
    private function log_policy_update($policy_update, $webhook_data, $status = 'applied') {
        // Extract violation count from webhook data (violations array)
        $violation_count = 0;
        if (isset($webhook_data['violations']) && is_array($webhook_data['violations'])) {
            $violation_count = count($webhook_data['violations']);
        } elseif (isset($policy_update['violation_count'])) {
            $violation_count = $policy_update['violation_count'];
        }
        
        $log_entry = array(
            'timestamp' => current_time('mysql'),
            'action' => $policy_update['action'] ?? 'unknown',
            'reason' => $policy_update['reason'] ?? 'unknown',
            'confidence_score' => $policy_update['confidence_score'] ?? 0,
            'violation_count' => $violation_count,
            'old_policy' => $policy_update['old_policy'] ?? '',
            'new_policy' => $policy_update['new_policy'] ?? '',
            'changes' => $policy_update['changes'] ?? array(),
            'status' => $status, // 'applied' or 'skipped'
            'webhook_data' => $webhook_data
        );
        
        // Store in WordPress options for now (could be moved to a custom table later)
        $updates = get_option('metricpoints_csp_policy_updates', array());
        $updates[] = $log_entry;
        
        // Keep only the last 50 updates
        if (count($updates) > 50) {
            $updates = array_slice($updates, -50);
        }
        
        update_option('metricpoints_csp_policy_updates', $updates);
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('MetricPoints WordPress Webhook: Policy updated - ' . json_encode($log_entry));
        }
        
        // Create backup before updating policy
        $this->create_policy_backup($policy_update['old_policy'] ?? '', $policy_update['new_policy'] ?? '');
    }
    
    private function create_policy_backup($old_policy, $new_policy) {
        $backups = get_option('metricpoints_csp_policy_backups', array());
        
        $backup = array(
            'timestamp' => current_time('mysql'),
            'old_policy' => $old_policy,
            'new_policy' => $new_policy,
            'backup_id' => uniqid('backup_', true)
        );
        
        array_unshift($backups, $backup); // Add to beginning
        
        // Keep only configured number of backups
        $backup_limit = isset($this->options['backup_limit']) ? max(10, intval($this->options['backup_limit'])) : 20;
        if (count($backups) > $backup_limit) {
            $backups = array_slice($backups, 0, $backup_limit);
        }
        
        update_option('metricpoints_csp_policy_backups', $backups);
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('MetricPoints CSP: Policy backup created - ' . $backup['backup_id']);
        }
    }
    
    public function handle_rollback_policy() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_die('Security check failed');
        }
        
        $backup_id = sanitize_text_field($_POST['backup_id'] ?? '');
        
        if (empty($backup_id)) {
            wp_send_json_error('No backup ID provided');
            return;
        }
        
        $backups = get_option('metricpoints_csp_policy_backups', array());
        $backup = null;
        
        foreach ($backups as $b) {
            if ($b['backup_id'] === $backup_id) {
                $backup = $b;
                break;
            }
        }
        
        if (!$backup) {
            wp_send_json_error('Backup not found');
            return;
        }
        
        // Rollback to the old policy
        $this->options['csp_policy'] = $backup['old_policy'];
        update_option('metricpoints_csp_manager_options', $this->options);
        
        // Log the rollback
        $this->log_policy_update(array(
            'action' => 'rollback',
            'reason' => 'manual_rollback',
            'old_policy' => $backup['new_policy'],
            'new_policy' => $backup['old_policy'],
            'confidence_score' => 1.0,
            'violation_count' => 0
        ), array('rollback' => true, 'backup_id' => $backup_id));
        
        wp_send_json_success(array(
            'message' => 'Policy rolled back successfully',
            'restored_policy' => $backup['old_policy'],
            'timestamp' => current_time('mysql')
        ));
    }
    
    public function handle_force_update() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_die('Security check failed');
        }
        
        // Simulate a policy update (in real implementation, this would call MetricPoints API)
        $current_policy = $this->options['csp_policy'] ?? '';
        $updated_policy = $this->simulate_policy_improvement($current_policy);
        
        if ($updated_policy !== $current_policy) {
            // Create backup
            $this->create_policy_backup($current_policy, $updated_policy);
            
            // Update policy
            $this->options['csp_policy'] = $updated_policy;
            update_option('metricpoints_csp_manager_options', $this->options);
            
            // Log the update
            $this->log_policy_update(array(
                'action' => 'force_update',
                'reason' => 'manual_force_update',
                'old_policy' => $current_policy,
                'new_policy' => $updated_policy,
                'confidence_score' => 0.8,
                'violation_count' => 0
            ), array('force_update' => true));
            
            wp_send_json_success(array(
                'message' => 'Policy force updated successfully',
                'updated_policy' => $updated_policy,
                'timestamp' => current_time('mysql')
            ));
        } else {
            wp_send_json_success(array(
                'message' => 'Policy is already up to date',
                'updated_policy' => $current_policy,
                'timestamp' => current_time('mysql')
            ));
        }
    }
    
    public function handle_reset_learning() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_die('Security check failed');
        }
        
        // Clear all learning data
        delete_option('metricpoints_csp_policy_updates');
        delete_option('metricpoints_csp_policy_backups');
        delete_option('metricpoints_csp_violations');
        delete_option('metricpoints_csp_last_update');
        delete_option('metricpoints_csp_policy_updates');
        
        // Reset counters
        update_option('metricpoints_csp_policy_updates', 0);
        
        wp_send_json_success(array(
            'message' => 'Learning data reset successfully',
            'timestamp' => current_time('mysql')
        ));
    }
    
    public function handle_get_backups() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_die('Security check failed');
        }
        
        $backups = get_option('metricpoints_csp_policy_backups', array());
        $show_all = isset($_POST['show_all']) && $_POST['show_all'] == '1';
        
        if ($show_all) {
            // Return all backups with HTML for display
            $html = $this->generate_backup_table_html($backups, true);
            
            wp_send_json_success(array(
                'backups' => $backups,
                'count' => count($backups),
                'html' => $html
            ));
        } else {
            // Return just the backup data (for preview functionality)
            wp_send_json_success(array(
                'backups' => $backups,
                'count' => count($backups)
            ));
        }
    }
    
    /**
     * Generate backup table HTML
     */
    private function generate_backup_table_html($backups, $show_all = false) {
        if (empty($backups)) {
            return '<p style="color: #666; font-style: italic;">No policy backups available yet.</p>';
        }
        
        // Sort backups by timestamp (newest first)
        usort($backups, function($a, $b) {
            return strtotime($b['timestamp']) - strtotime($a['timestamp']);
        });
        
        // Limit to 5 if not showing all
        $display_backups = $show_all ? $backups : array_slice($backups, 0, 5);
        
        $html = '<div style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; border-radius: 4px;">';
        $html .= '<table class="wp-list-table widefat fixed striped" style="margin: 0;">';
        $html .= '<thead>';
        $html .= '<tr>';
        $html .= '<th style="padding: 8px;">Date</th>';
        $html .= '<th style="padding: 8px;">Reason</th>';
        $html .= '<th style="padding: 8px;">Actions</th>';
        $html .= '</tr>';
        $html .= '</thead>';
        $html .= '<tbody>';
        
        foreach ($display_backups as $index => $backup) {
            $date = date('M j, Y g:i A', strtotime($backup['timestamp']));
            $reason = esc_html($backup['reason']);
            
            $html .= '<tr>';
            $html .= '<td style="padding: 8px;">' . $date . '</td>';
            $html .= '<td style="padding: 8px;">' . $reason . '</td>';
            $html .= '<td style="padding: 8px;">';
            $html .= '<button class="button button-small" onclick="previewBackup(' . $index . ')" title="Preview Policy">👁️ Preview</button> ';
            $html .= '<button class="button button-small" onclick="restoreBackup(' . $index . ')" style="background: #28a745; color: white;" title="Restore This Policy">↩️ Restore</button>';
            $html .= '</td>';
            $html .= '</tr>';
        }
        
        $html .= '</tbody></table>';
        $html .= '</div>';
        
        // Get backup limit for display
        $backup_limit = isset($this->options['backup_limit']) ? max(10, intval($this->options['backup_limit'])) : 20;
        
        if (!$show_all && count($backups) > 5) {
            $html .= '<div style="margin-top: 10px; text-align: center;">';
            $html .= '<button id="show-more-backups" class="button button-secondary" style="margin-right: 10px;">📋 Show All Backups (' . count($backups) . ')</button>';
            $html .= '<button id="show-recent-backups" class="button button-secondary" style="display: none;">📋 Show Recent Only (5)</button>';
            $html .= '</div>';
            $html .= '<p style="color: #666; font-size: 12px; margin-top: 5px; text-align: center;">Showing latest 5 backups. ' . (count($backups) - 5) . ' more available. (Limit: ' . $backup_limit . ')</p>';
        } elseif ($show_all) {
            $html .= '<div style="margin-top: 10px; text-align: center;">';
            $html .= '<button id="show-recent-backups" class="button button-secondary">📋 Show Recent Only (5)</button>';
            $html .= '</div>';
            $html .= '<p style="color: #666; font-size: 12px; margin-top: 5px; text-align: center;">Showing all ' . count($backups) . ' backups. (Limit: ' . $backup_limit . ')</p>';
        } else {
            // Show limit info even when showing all backups
            $html .= '<p style="color: #666; font-size: 12px; margin-top: 5px; text-align: center;">Backup limit: ' . $backup_limit . ' (Current: ' . count($backups) . ')</p>';
        }
        
        return $html;
    }
    
    public function handle_analyze_violations() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_die('Security check failed');
        }
        
        // Get current violations
        $violations = get_option('metricpoints_csp_violations', array());
        $current_policy = $this->options['csp_policy'] ?? '';
        
        $analysis_results = array(
            'violations_found' => count($violations),
            'violations_analyzed' => 0,
            'recommendations' => array(),
            'policy_changes' => array(),
            'confidence_score' => 0.0
        );
        
        if (empty($violations)) {
            // No violations to analyze - suggest generating some test violations
            $analysis_results['message'] = 'No violations found to analyze. Try generating some test violations first using the "Generate CSP Violation" button.';
            $analysis_results['recommendations'][] = 'Generate test violations to see how the analysis works';
            $analysis_results['recommendations'][] = 'Make sure your CSP policy is active and blocking some resources';
            
            wp_send_json_success($analysis_results);
            return;
        }
        
        // Analyze violations
        $directive_violations = array();
        $blocked_uris = array();
        $document_uris = array();
        
        foreach ($violations as $violation) {
            $directive = $violation['violated_directive'] ?? 'unknown';
            $blocked_uri = $violation['blocked_uri'] ?? '';
            $document_uri = $violation['document_uri'] ?? '';
            
            if (!isset($directive_violations[$directive])) {
                $directive_violations[$directive] = 0;
            }
            $directive_violations[$directive]++;
            
            if (!in_array($blocked_uri, $blocked_uris)) {
                $blocked_uris[] = $blocked_uri;
            }
            
            if (!in_array($document_uri, $document_uris)) {
                $document_uris[] = $document_uri;
            }
            
            $analysis_results['violations_analyzed']++;
        }
        
        // Generate recommendations based on violations
        $recommendations = array();
        $policy_changes = array();
        
        foreach ($directive_violations as $directive => $count) {
            if ($count > 0) {
                $recommendations[] = "Found {$count} violations for '{$directive}' directive";
                
                // Suggest policy changes based on common patterns
                if ($directive === 'script-src' && strpos($current_policy, 'unsafe-inline') === false) {
                    $recommendations[] = "Consider adding 'unsafe-inline' to script-src for inline scripts";
                    $policy_changes[] = "Add 'unsafe-inline' to script-src directive";
                }
                
                if ($directive === 'style-src' && strpos($current_policy, 'unsafe-inline') === false) {
                    $recommendations[] = "Consider adding 'unsafe-inline' to style-src for inline styles";
                    $policy_changes[] = "Add 'unsafe-inline' to style-src directive";
                }
                
                if ($directive === 'img-src' && strpos($current_policy, 'data:') === false) {
                    $recommendations[] = "Consider adding 'data:' to img-src for data URIs";
                    $policy_changes[] = "Add 'data:' to img-src directive";
                }
            }
        }
        
        // Analyze blocked URIs
        $external_domains = array();
        foreach ($blocked_uris as $uri) {
            if (strpos($uri, 'http') === 0) {
                $parsed = parse_url($uri);
                if (isset($parsed['host'])) {
                    $domain = $parsed['host'];
                    if (!in_array($domain, $external_domains)) {
                        $external_domains[] = $domain;
                    }
                }
            }
        }
        
        if (!empty($external_domains)) {
            $recommendations[] = "Found external domains being blocked: " . implode(', ', array_slice($external_domains, 0, 5));
            if (count($external_domains) > 5) {
                $recommendations[] = "And " . (count($external_domains) - 5) . " more external domains";
            }
        }
        
        // Calculate confidence score based on analysis
        $confidence_score = min(0.9, 0.3 + (count($violations) * 0.1) + (count($recommendations) * 0.05));
        
        $analysis_results['recommendations'] = $recommendations;
        $analysis_results['policy_changes'] = $policy_changes;
        $analysis_results['confidence_score'] = $confidence_score;
        $analysis_results['directive_violations'] = $directive_violations;
        $analysis_results['blocked_uris'] = array_slice($blocked_uris, 0, 10); // Show first 10
        $analysis_results['external_domains'] = $external_domains;
        $analysis_results['message'] = "Analyzed {$analysis_results['violations_analyzed']} violations and generated " . count($recommendations) . " recommendations";
        
        // Log the analysis
        $this->log_policy_update(array(
            'action' => 'analyze_violations',
            'reason' => 'manual_analysis',
            'old_policy' => $current_policy,
            'new_policy' => $current_policy, // No change yet
            'confidence_score' => $confidence_score,
            'violation_count' => count($violations)
        ), array('analysis' => true, 'results' => $analysis_results));
        
        wp_send_json_success($analysis_results);
    }
    
    public function handle_activate_skipped_update() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_die('Security check failed');
        }
        
        $update_index = intval($_POST['update_index']);
        $updates = get_option('metricpoints_csp_policy_updates', array());
        
        if (!isset($updates[$update_index])) {
            wp_send_json_error('Update not found');
        }
        
        $update = $updates[$update_index];
        
        if ($update['status'] !== 'skipped' || empty($update['new_policy'])) {
            wp_send_json_error('Update cannot be activated');
        }
        
        // Validate the new policy
        $validation = $this->validate_csp_policy($update['new_policy']);
        if (!$validation['valid']) {
            wp_send_json_error('Invalid policy: ' . implode(', ', $validation['errors']));
        }
        
        // Create backup before updating
        $this->create_policy_backup($this->options['csp_policy'], $update['new_policy']);
        
        // Update the policy
        $this->options['csp_policy'] = $update['new_policy'];
        update_option('metricpoints_csp_manager_options', $this->options);
        
        // Update the status in the log
        $updates[$update_index]['status'] = 'applied';
        $updates[$update_index]['applied_timestamp'] = current_time('mysql');
        update_option('metricpoints_csp_policy_updates', $updates);
        
        wp_send_json_success(array(
            'message' => 'Policy update activated successfully',
            'updated_policy' => $update['new_policy']
        ));
    }
    
    public function handle_test_webhook() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints: Nonce verification failed');
            }
            wp_die('Security check failed');
        }
        
        $api_key = $this->options['api_key'] ?? '';
        
        if (empty($api_key)) {
            wp_send_json_error('API key not configured');
        }
        
        $webhook_url = $this->get_self_healing_webhook_url();
        
        // Create realistic test payload
        $test_payload = array(
            'webhook_type' => 'wordpress_csp_update',
            'timestamp' => current_time('c'),
            'site_info' => array(
                'site_url' => home_url(),
                'plugin_version' => '1.0.0',
                'wordpress_version' => get_bloginfo('version')
            ),
            'project' => array(
                'id' => 'test_project_' . time(),
                'name' => 'Test Project - WordPress Plugin',
                'domain' => parse_url(home_url(), PHP_URL_HOST),
                'team_id' => 1
            ),
            'policy_update' => array(
                'action' => 'test_update',
                'reason' => 'webhook_test',
                'old_policy' => 'default-src \'self\'; script-src \'self\';',
                'new_policy' => 'default-src \'self\'; script-src \'self\' \'unsafe-inline\' https://cdn.example.com; style-src \'self\' \'unsafe-inline\';',
                'changes' => array(
                    array(
                        'directive' => 'script-src',
                        'old_value' => '\'self\'',
                        'new_value' => '\'self\' \'unsafe-inline\' https://cdn.example.com',
                        'reason' => 'Added unsafe-inline and CDN for testing'
                    ),
                    array(
                        'directive' => 'style-src',
                        'old_value' => '',
                        'new_value' => '\'self\' \'unsafe-inline\'',
                        'reason' => 'Added style-src directive for testing'
                    )
                ),
                'confidence_score' => 0.92,
                'violation_count' => 3
            ),
            'violations' => array(
                array(
                    'directive' => 'script-src',
                    'blocked_uri' => 'inline',
                    'violated_directive' => 'script-src',
                    'document_uri' => home_url(),
                    'timestamp' => current_time('c'),
                    'source_file' => 'test-script.js',
                    'line_number' => 15
                ),
                array(
                    'directive' => 'script-src',
                    'blocked_uri' => 'https://cdn.example.com/script.js',
                    'violated_directive' => 'script-src',
                    'document_uri' => home_url(),
                    'timestamp' => current_time('c'),
                    'source_file' => 'index.html',
                    'line_number' => 8
                ),
                array(
                    'directive' => 'style-src',
                    'blocked_uri' => 'inline',
                    'violated_directive' => 'style-src',
                    'document_uri' => home_url(),
                    'timestamp' => current_time('c'),
                    'source_file' => 'styles.css',
                    'line_number' => 22
                )
            )
        );
        
        // Test connectivity first - try multiple host addresses (only in debug mode)
        $hosts = ['host.docker.internal', '172.17.0.1', '172.18.0.1'];
        $connected = false;
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            foreach ($hosts as $host) {
                $test_response = wp_remote_get("http://$host:8000/api/test-connectivity", array('timeout' => 3));
                if (!is_wp_error($test_response)) {
                    $test_code = wp_remote_retrieve_response_code($test_response);
                    if ($test_code === 200) {
                        $connected = true;
                        break;
                    }
                }
            }
        } else {
            // In production, skip connectivity test and proceed directly
            $connected = true;
        }
        
        if (!$connected) {
            wp_send_json_error('Cannot connect to server. Please check your configuration.');
        }
        
        $response = wp_remote_post($webhook_url, array(
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-MetricPoints-API-Key' => $api_key,
                'User-Agent' => 'MetricPoints-WordPress-Plugin/1.0.0'
            ),
            'body' => json_encode($test_payload),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints: Webhook request failed: ' . $response->get_error_message());
            }
            wp_send_json_error('Failed to send webhook: ' . $response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        
        if ($response_code >= 200 && $response_code < 300) {
            wp_send_json_success(array(
                'message' => 'Webhook test successful',
                'response_code' => $response_code,
                'response' => $response_data,
                'webhook_url' => $webhook_url
            ));
        } else {
            wp_send_json_error(array(
                'message' => 'Webhook test failed',
                'response_code' => $response_code,
                'response' => $response_data,
                'webhook_url' => $webhook_url
            ));
        }
    }
    
    private function simulate_policy_improvement($current_policy) {
        // Simple simulation - in real implementation, this would call MetricPoints API
        if (empty($current_policy)) {
            return "default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline';";
        }
        
        // Add some common improvements
        $improvements = array(
            "img-src 'self' data: https:;",
            "font-src 'self' https:;",
            "connect-src 'self' https:;",
            "frame-ancestors 'none';",
            "base-uri 'self';"
        );
        
        $new_policy = $current_policy;
        foreach ($improvements as $improvement) {
            if (strpos($new_policy, $improvement) === false) {
                $new_policy = rtrim($new_policy, ';') . '; ' . $improvement;
                break; // Add one improvement at a time
            }
        }
        
        return $new_policy;
    }
    
    private function display_webhook_history() {
        $updates = get_option('metricpoints_csp_policy_updates', array());
        
        if (empty($updates)) {
            echo '<p style="color: #666; font-style: italic;">No policy updates received yet.</p>';
            return;
        }
        
        // Show only the last 10 updates
        $recent_updates = array_slice($updates, -10);
        $recent_updates = array_reverse($recent_updates); // Show newest first
        
        echo '<div style="max-height: 300px; overflow-y: auto; border: 1px solid #ddd; border-radius: 4px; padding: 10px;">';
        
        foreach ($recent_updates as $index => $update) {
            $timestamp = $update['timestamp'] ?? 'Unknown';
            $action = $update['action'] ?? 'unknown';
            $reason = $update['reason'] ?? 'unknown';
            $confidence = $update['confidence_score'] ?? 0;
            $violation_count = $update['violation_count'] ?? 0;
            $status = $update['status'] ?? 'applied';
            
            echo '<div style="border-bottom: 1px solid #eee; padding: 10px 0; margin-bottom: 10px;">';
            echo '<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 5px;">';
            
            // Status indicator
            if ($status === 'skipped') {
                echo '<strong style="color: #ff9800;">⏸️ ' . esc_html(ucfirst($action)) . ' (Skipped)</strong>';
            } else {
                echo '<strong style="color: #007cba;">✅ ' . esc_html(ucfirst($action)) . '</strong>';
            }
            
            echo '<span style="color: #666; font-size: 12px;">' . esc_html($timestamp) . '</span>';
            echo '</div>';
            
            echo '<p style="margin: 5px 0; color: #666; font-size: 14px;">';
            echo '<strong>Reason:</strong> ' . esc_html($reason) . '<br>';
            echo '<strong>Confidence:</strong> ' . round($confidence * 100, 1) . '% | ';
            echo '<strong>Violations:</strong> ' . esc_html($violation_count);
            echo '</p>';
            
            // Manual activation button for skipped updates
            if ($status === 'skipped' && !empty($update['new_policy'])) {
                echo '<div style="margin: 10px 0;">';
                echo '<button class="button button-small" onclick="activateSkippedUpdate(' . $index . ')" style="background: #28a745; color: white;">🚀 Activate This Update</button>';
                echo '<span style="margin-left: 10px; color: #666; font-size: 12px;">This update was skipped due to low confidence but can be manually activated.</span>';
                echo '</div>';
            }
            
            if (!empty($update['changes'])) {
                echo '<details style="margin-top: 5px;">';
                echo '<summary style="cursor: pointer; color: #007cba; font-size: 12px;">View Changes</summary>';
                echo '<ul style="margin: 5px 0 0 20px; font-size: 12px; color: #666;">';
                foreach ($update['changes'] as $change) {
                    echo '<li><strong>' . esc_html($change['directive'] ?? 'unknown') . ':</strong> ';
                    echo esc_html($change['old_value'] ?? '') . ' → ';
                    echo esc_html($change['new_value'] ?? '') . '</li>';
                }
                echo '</ul>';
                echo '</details>';
            }
            
            echo '</div>';
        }
        
        echo '</div>';
        
        if (count($updates) > 10) {
            echo '<p style="text-align: center; color: #666; font-size: 12px; margin-top: 10px;">';
            echo 'Showing last 10 of ' . count($updates) . ' updates';
            echo '</p>';
        }
    }
    
    public function enqueue_scripts() {
        // Always enqueue admin scripts for the admin interface
        if (is_admin()) {
            wp_enqueue_script(
                'metricpoints-csp-manager',
                METRICPOINTS_CSP_MANAGER_PLUGIN_URL . 'js/csp-manager.js',
                array(),
                METRICPOINTS_CSP_MANAGER_VERSION,
                true
            );
            
            wp_localize_script('metricpoints-csp-manager', 'metricpointsCspManager', array(
                'reportUri' => $this->options['report_uri'] ?? admin_url('admin-ajax.php?action=metricpoints_csp_violation'),
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('metricpoints_csp_manager_nonce')
            ));
        }
        
        // Only enqueue frontend scripts when plugin is enabled
        if (empty($this->options['enabled'])) {
            return;
        }
        
        // Frontend scripts would go here if needed
    }
    
    public function handle_csp_violation() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_die('Security check failed');
        }
        
        $violation_data = json_decode(file_get_contents('php://input'), true);
        
        if ($violation_data) {
            $this->process_csp_violation($violation_data);
        }
        
        wp_die();
    }
    
    private function process_csp_violation($violation_data) {
        $violation = array(
            'timestamp' => current_time('mysql'),
            'violated_directive' => sanitize_text_field($violation_data['violated-directive'] ?? ''),
            'blocked_uri' => esc_url_raw($violation_data['blocked-uri'] ?? ''),
            'document_uri' => esc_url_raw($violation_data['document-uri'] ?? ''),
            'source_file' => sanitize_text_field($violation_data['source-file'] ?? ''),
            'line_number' => intval($violation_data['line-number'] ?? 0),
            'column_number' => intval($violation_data['column-number'] ?? 0),
            'status_code' => intval($violation_data['status-code'] ?? 0),
            'user_agent' => sanitize_text_field($violation_data['user-agent'] ?? ''),
            'raw_data' => $violation_data
        );
        
        // Store violation
        $this->csp_violations[] = $violation;
        
        // Keep only last 100 violations
        if (count($this->csp_violations) > 100) {
            $this->csp_violations = array_slice($this->csp_violations, -100);
        }
        
        update_option('metricpoints_csp_violations', $this->csp_violations);
        
        // Send to MetricPoints API
        $this->send_to_metricpoints($violation);
        
        // Self-healing logic
        if (!empty($this->options['self_healing'])) {
            $this->analyze_and_update_policy($violation);
        }
    }
    
    private function send_to_metricpoints($violation) {
        if (empty($this->options['api_key'])) {
            return;
        }
        
        $response = wp_remote_post('https://metricpoints.com/api/csp-reports', array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->options['api_key']
            ),
            'body' => json_encode($violation),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP Manager: Failed to send violation report - ' . $response->get_error_message());
            }
        }
    }
    
    private function analyze_and_update_policy($violation) {
        $current_policy = $this->get_current_csp_policy();
        $violated_directive = $violation['violated_directive'];
        $blocked_uri = $violation['blocked_uri'];
        $document_uri = $violation['document_uri'];
        
        // Extract domain from blocked URI
        $domain = parse_url($blocked_uri, PHP_URL_HOST);
        if (!$domain) {
            return;
        }
        
        // Store violation for pattern analysis
        $this->store_violation_for_analysis($violation);
        
        // Analyze violation patterns
        $analysis = $this->analyze_violation_patterns($violated_directive, $domain);
        
        // Apply intelligent policy updates based on analysis
        $updated_policy = $this->apply_intelligent_policy_update($current_policy, $analysis);
        
        if ($updated_policy !== $current_policy) {
            // Create backup before updating
            $this->create_policy_backup($current_policy, $updated_policy);
            
            $this->options['csp_policy'] = $updated_policy;
            update_option('metricpoints_csp_manager_options', $this->options);
            $this->increment_policy_update_count();
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log("MetricPoints CSP Manager: AI updated policy for {$violated_directive} - Added {$domain} (Confidence: {$analysis['confidence']}%)");
            }
        }
    }
    
    private function store_violation_for_analysis($violation) {
        $violation_data = array(
            'timestamp' => current_time('mysql'),
            'directive' => $violation['violated_directive'],
            'domain' => parse_url($violation['blocked_uri'], PHP_URL_HOST),
            'path' => parse_url($violation['blocked_uri'], PHP_URL_PATH),
            'scheme' => parse_url($violation['blocked_uri'], PHP_URL_SCHEME),
            'document_uri' => $violation['document_uri'],
            'user_agent' => $violation['user_agent'],
            'raw_violation' => $violation
        );
        
        $analysis_data = get_option('metricpoints_csp_analysis_data', array());
        $analysis_data[] = $violation_data;
        
        // Keep only last 1000 violations for analysis
        if (count($analysis_data) > 1000) {
            $analysis_data = array_slice($analysis_data, -1000);
        }
        
        update_option('metricpoints_csp_analysis_data', $analysis_data);
    }
    
    private function analyze_violation_patterns($directive, $domain) {
        $analysis_data = get_option('metricpoints_csp_analysis_data', array());
        $domain_violations = array_filter($analysis_data, function($v) use ($domain) {
            return $v['domain'] === $domain;
        });
        
        $directive_violations = array_filter($analysis_data, function($v) use ($directive) {
            return $v['directive'] === $directive;
        });
        
        // Calculate confidence based on frequency and patterns
        $confidence = $this->calculate_trust_confidence($domain, $domain_violations, $directive_violations);
        
        // Determine if this is a legitimate resource that should be allowed
        $is_legitimate = $this->is_legitimate_resource($domain, $directive, $analysis_data);
        
        return array(
            'domain' => $domain,
            'directive' => $directive,
            'confidence' => $confidence,
            'is_legitimate' => $is_legitimate,
            'violation_count' => count($domain_violations),
            'recommendation' => $this->get_recommendation($directive, $domain, $confidence, $is_legitimate)
        );
    }
    
    private function calculate_trust_confidence($domain, $domain_violations, $directive_violations) {
        $base_confidence = 50; // Start with neutral confidence
        
        // Increase confidence based on frequency
        $frequency_factor = min(count($domain_violations) * 5, 30);
        
        // Increase confidence for well-known domains
        $well_known_domains = array(
            'googleapis.com', 'gstatic.com', 'google.com', 'googleusercontent.com',
            'cdnjs.cloudflare.com', 'cdn.jsdelivr.net', 'unpkg.com',
            'fonts.googleapis.com', 'fonts.gstatic.com',
            'ajax.googleapis.com', 'maps.googleapis.com'
        );
        
        $domain_factor = 0;
        foreach ($well_known_domains as $known_domain) {
            if (strpos($domain, $known_domain) !== false) {
                $domain_factor = 25;
                break;
            }
        }
        
        // Increase confidence for HTTPS
        $scheme_factor = 0;
        foreach ($domain_violations as $violation) {
            if (isset($violation['scheme']) && $violation['scheme'] === 'https') {
                $scheme_factor = 10;
                break;
            }
        }
        
        return min($base_confidence + $frequency_factor + $domain_factor + $scheme_factor, 95);
    }
    
    private function is_legitimate_resource($domain, $directive, $analysis_data) {
        // Check if this domain appears frequently across different pages
        $domain_usage = array_filter($analysis_data, function($v) use ($domain) {
            return $v['domain'] === $domain;
        });
        
        $unique_pages = array_unique(array_column($domain_usage, 'document_uri'));
        
        // If domain appears on multiple pages, it's likely legitimate
        if (count($unique_pages) > 1) {
            return true;
        }
        
        // Check for common legitimate patterns
        $legitimate_patterns = array(
            '/wp-content/', '/wp-includes/', '/wp-admin/',
            '/assets/', '/css/', '/js/', '/images/', '/img/',
            '/fonts/', '/static/', '/public/'
        );
        
        foreach ($domain_usage as $usage) {
            if (isset($usage['path'])) {
                foreach ($legitimate_patterns as $pattern) {
                    if (strpos($usage['path'], $pattern) !== false) {
                        return true;
                    }
                }
            }
        }
        
        return false;
    }
    
    private function get_recommendation($directive, $domain, $confidence, $is_legitimate) {
        if (!$is_legitimate) {
            return 'block';
        }
        
        if ($confidence >= 80) {
            return 'allow';
        } elseif ($confidence >= 60) {
            return 'allow_with_monitoring';
        } else {
            return 'review_manually';
        }
    }
    
    private function apply_intelligent_policy_update($policy, $analysis) {
        if ($analysis['recommendation'] === 'block' || $analysis['recommendation'] === 'review_manually') {
            return $policy; // Don't update
        }
        
        if ($analysis['confidence'] < 60) {
            return $policy; // Not confident enough
        }
        
        return $this->update_csp_directive($policy, $analysis['directive'], $analysis['domain']);
    }
    
    
    private function update_csp_directive($policy, $directive, $domain) {
        // Simple policy update logic
        // In a real implementation, this would be more sophisticated
        
        $directive_pattern = '/(' . preg_quote($directive) . '\s+[^;]+)/';
        
        if (preg_match($directive_pattern, $policy, $matches)) {
            $current_directive = $matches[1];
            
            // Check if domain is already included
            if (strpos($current_directive, $domain) === false) {
                // Add domain to directive
                $updated_directive = rtrim($current_directive, ';') . ' ' . $domain . ';';
                $policy = str_replace($current_directive, $updated_directive, $policy);
            }
        } else {
            // Add new directive
            $policy .= $directive . ' ' . $domain . '; ';
        }
        
        return $policy;
    }
    
    private function get_current_csp_policy() {
        $policy = $this->options['csp_policy'] ?? $this->get_default_csp_policy();
        
        // Remove comments (lines starting with #)
        $policy = preg_replace('/^#.*$/m', '', $policy);
        $policy = preg_replace('/\s*#.*$/m', '', $policy);
        
        // Clean up extra whitespace and newlines
        $policy = preg_replace('/\s+/', ' ', $policy);
        $policy = trim($policy);
        
        // Generate reporting URI based on API key
        $api_key = $this->options['api_key'] ?? '';
        if (!empty($api_key)) {
            $report_uri = "https://metricpoints.com/api/csp-reports/{$api_key}";
        } else {
            $report_uri = $this->options['report_uri'] ?? admin_url('admin-ajax.php?action=metricpoints_csp_violation');
        }
        
        // Add report-uri and report-to directives if not already present
        if (strpos($policy, 'report-uri') === false) {
            $policy .= " report-uri {$report_uri};";
        } else {
            $policy = str_replace('{REPORT_URI}', $report_uri, $policy);
        }
        
        // Add report-to directive for modern browsers
        if (strpos($policy, 'report-to') === false) {
            $policy .= " report-to csp-endpoint;";
        }
        
        return $policy;
    }
    
    private function get_default_csp_policy() {
        return "default-src 'self'; script-src 'self' 'unsafe-inline' 'unsafe-eval'; style-src 'self' 'unsafe-inline'; img-src 'self' data: https:; font-src 'self' data:; connect-src 'self';";
    }
    
    private function display_violations() {
        if (empty($this->csp_violations)) {
            echo '<p>No violations detected yet.</p>';
            return;
        }
        
        $recent_violations = array_slice($this->csp_violations, -10);
        
        echo '<table class="widefat">';
        echo '<thead><tr><th>Time</th><th>Directive</th><th>Blocked URI</th><th>Document URI</th></tr></thead>';
        echo '<tbody>';
        
        foreach (array_reverse($recent_violations) as $violation) {
            echo '<tr>';
            echo '<td>' . esc_html($violation['timestamp']) . '</td>';
            echo '<td>' . esc_html($violation['violated_directive']) . '</td>';
            echo '<td>' . esc_html($violation['blocked_uri']) . '</td>';
            echo '<td>' . esc_html($violation['document_uri']) . '</td>';
            echo '</tr>';
        }
        
        echo '</tbody></table>';
    }
    
    private function get_policy_update_count() {
        return get_option('metricpoints_csp_policy_updates', 0);
    }
    
    private function get_last_policy_update() {
        $last_update = get_option('metricpoints_csp_last_update', '');
        if (empty($last_update)) {
            return 'Never';
        }
        return date('M j, Y g:i A', strtotime($last_update));
    }
    
    private function increment_policy_update_count() {
        $count = $this->get_policy_update_count();
        update_option('metricpoints_csp_policy_updates', $count + 1);
        update_option('metricpoints_csp_last_update', current_time('mysql'));
    }
    
    private function display_csp_builder_marketing() {
        // Always show the colorful call-to-action for all users
            echo '<div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px; margin: 15px 0; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">';
            echo '<h4 style="margin: 0 0 15px 0; color: white;">🎯 FREE CSP Policy Builder from MetricPoints</h4>';
            echo '<p style="margin: 0 0 15px 0; opacity: 0.9;">Create bulletproof Content Security Policies in minutes, not hours!</p>';
            echo '<div style="display: flex; gap: 15px; flex-wrap: wrap; align-items: center;">';
            echo '<div style="flex: 1; min-width: 200px;">';
            echo '<ul style="margin: 0; padding-left: 20px; font-size: 14px; opacity: 0.9;">';
            echo '<li>✅ Visual policy builder</li>';
            echo '<li>✅ Pre-built templates</li>';
            echo '<li>✅ Real-time validation</li>';
            echo '<li>✅ Export ready-to-use policies</li>';
            echo '</ul>';
            echo '</div>';
            echo '<div style="text-align: center;">';
            echo '<a href="https://metricpoints.com/csp-policy-builder" target="_blank" style="background: white; color: #667eea; padding: 12px 24px; border-radius: 25px; text-decoration: none; font-weight: bold; display: inline-block; box-shadow: 0 2px 10px rgba(0,0,0,0.2);">🚀 Start Building FREE</a>';
            echo '</div>';
            echo '</div>';
        echo '</div>';
        
        // Check if user has error tracking subscription for additional upsell
        $has_error_tracking = $this->has_error_tracking_subscription();
        if (!$has_error_tracking) {
            // Additional call to action for users without error tracking
            echo '<div style="background: #f8f9fa; padding: 15px; border-radius: 5px; margin: 10px 0; border-left: 4px solid #007cba;">';
            echo '<p style="margin: 0 0 10px 0;"><strong>🚀 Complete Your Security Stack!</strong></p>';
            echo '<p style="margin: 0 0 10px 0; color: #666;">You\'re using our CSP Manager - the perfect companion to our <a href="https://metricpoints.com/error-tracking" target="_blank" style="color: #0073aa; font-weight: bold;">Error Tracking</a> service!</p>';
            echo '<p style="margin: 0 0 10px 0; color: #666;">Get comprehensive security monitoring with both CSP violation tracking AND JavaScript error tracking in one powerful platform.</p>';
            echo '<p style="margin: 0;"><a href="https://metricpoints.com/error-tracking" target="_blank" style="color: #0073aa; font-weight: bold;">📊 Add Error Tracking</a></p>';
            echo '</div>';
        }
    }
    
    private function is_user_logged_in() {
        // Check if user has a valid API key (indicates they're logged in)
        $api_key = isset($this->options['api_key']) ? $this->options['api_key'] : '';
        return !empty($api_key);
    }
    
    private function has_error_tracking_subscription() {
        // Check if error tracking plugin is active and configured
        $error_tracker_options = get_option('metricpoints_error_tracker_options', array());
        $has_error_tracking = !empty($error_tracker_options['enabled']) && !empty($error_tracker_options['api_key']);
        
        // Also check if the error tracker plugin is active
        $is_plugin_active = is_plugin_active('metricpoints-error-tracker/metricpoints-error-tracker.php');
        
        return $has_error_tracking && $is_plugin_active;
    }
    
    private function get_self_healing_webhook_url() {
        $api_key = isset($this->options['api_key']) ? $this->options['api_key'] : '';
        
        if (empty($api_key)) {
            return 'https://metricpoints.com/api/csp-self-healing/webhook/{your-api-key}';
        }
        
        // Check if we're in local development mode
        $is_local_dev = defined('WP_DEBUG') && WP_DEBUG && 
                        (strpos(home_url(), '127.0.0.1') !== false || 
                         strpos(home_url(), 'localhost') !== false ||
                         strpos(home_url(), '.local') !== false);
        
        if ($is_local_dev) {
            // Use local Laravel development server
            // WordPress is in Docker, try multiple host addresses
            $hosts = [
                'host.docker.internal',
                '172.17.0.1',  // Default Docker bridge gateway
                '172.18.0.1',  // Alternative Docker bridge
                'host-gateway' // Docker 20.10+ alias
            ];
            
            // For now, try host.docker.internal first
            return "http://host.docker.internal:8000/api/csp-self-healing/webhook/{$api_key}";
        }
        
        // Production URL
        return "https://metricpoints.com/api/csp-self-healing/webhook/{$api_key}";
    }
    
    private function display_policy_backups() {
        $backups = get_option('metricpoints_csp_policy_backups', array());
        echo $this->generate_backup_table_html($backups, false);
    }
    
    /**
     * Handle WordPress registration AJAX request
     */
    public function handle_register() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $name = sanitize_text_field($_POST['name'] ?? '');
        $email = sanitize_email($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $domain = sanitize_text_field($_POST['domain'] ?? '');
        
        if (empty($name) || empty($email) || empty($password)) {
            wp_send_json_error('Name, email, and password are required');
        }
        
        if (strlen($password) < 8) {
            wp_send_json_error('Password must be at least 8 characters');
        }
        
        // Determine API base URL
        $is_local_dev = defined('WP_DEBUG') && WP_DEBUG && 
                        (strpos(home_url(), '127.0.0.1') !== false || 
                         strpos(home_url(), 'localhost') !== false ||
                         strpos(home_url(), '.local') !== false);
        
        if ($is_local_dev) {
            $api_base = 'http://host.docker.internal:8000/api';
        } else {
            $api_base = 'https://metricpoints.com/api';
        }
        
        $register_url = $api_base . '/v1/wordpress/register';
        
        $response = wp_remote_post($register_url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'MetricPoints-WordPress-Plugin/' . METRICPOINTS_CSP_MANAGER_VERSION
            ),
            'body' => json_encode(array(
                'name' => $name,
                'email' => $email,
                'password' => $password,
                'domain' => $domain ?: parse_url(home_url(), PHP_URL_HOST),
                'wordpress_site_url' => home_url(),
            ))
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error('Failed to connect to MetricPoints: ' . $response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        
        if ($response_code === 201 && isset($response_data['success']) && $response_data['success']) {
            // Store API key in options
            $this->options['api_key'] = $response_data['project']['api_key'] ?? '';
            $this->options['project_info'] = $response_data['project'] ?? null;
            $this->options['connection_status'] = 'connected';
            $this->options['connection_verified_at'] = current_time('mysql');
            update_option('metricpoints_csp_manager_options', $this->options);
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP Manager: User registered successfully via WordPress plugin - Email: ' . $email . ', Project ID: ' . ($response_data['project']['id'] ?? 'unknown'));
            }
            
            wp_send_json_success(array(
                'message' => 'Account created successfully!',
                'api_key' => $response_data['project']['api_key'] ?? '',
                'project' => $response_data['project'] ?? null,
            ));
        } else {
            $error_message = $response_data['error'] ?? $response_data['message'] ?? 'Registration failed';
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP Manager: Registration failed - Email: ' . $email . ', Error: ' . $error_message . ', Response Code: ' . $response_code);
            }
            
            wp_send_json_error($error_message);
        }
    }
    
    /**
     * Handle save headers AJAX request
     */
    public function handle_save_headers() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        if (empty($this->options['api_key'])) {
            wp_send_json_error('API key not configured');
        }
        
        // Get current headers
        $headers = $this->get_current_headers();
        
        // Get project info
        $project_info = $this->options['project_info'] ?? null;
        if (!$project_info || !isset($project_info['id'])) {
            wp_send_json_error('Project information not available');
        }
        
        // Determine API base URL
        $is_local_dev = defined('WP_DEBUG') && WP_DEBUG && 
                        (strpos(home_url(), '127.0.0.1') !== false || 
                         strpos(home_url(), 'localhost') !== false ||
                         strpos(home_url(), '.local') !== false);
        
        if ($is_local_dev) {
            $api_base = 'http://host.docker.internal:8000/api';
        } else {
            $api_base = 'https://metricpoints.com/api';
        }
        
        $save_url = $api_base . '/v1/csp-projects/' . $project_info['id'] . '/header-history';
        
        $response = wp_remote_post($save_url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $this->options['api_key'],
                'User-Agent' => 'MetricPoints-WordPress-Plugin/' . METRICPOINTS_CSP_MANAGER_VERSION
            ),
            'body' => json_encode(array(
                'headers' => $headers,
                'csp_policy' => $this->get_current_csp_policy(),
                'site_url' => home_url(),
            ))
        ));
        
        if (is_wp_error($response)) {
            $error_msg = 'Failed to connect to MetricPoints: ' . $response->get_error_message();
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP Manager: Failed to save headers (AJAX) - ' . $error_msg . ', Project ID: ' . ($project_info['id'] ?? 'unknown'));
            }
            wp_send_json_error($error_msg);
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        
        if ($response_code === 201 && isset($response_data['success']) && $response_data['success']) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP Manager: Headers saved successfully (AJAX) - Project ID: ' . ($project_info['id'] ?? 'unknown') . ', Headers count: ' . count($headers));
            }
            wp_send_json_success(array(
                'message' => 'Headers saved successfully',
            ));
        } else {
            $error_message = $response_data['error'] ?? 'Failed to save headers';
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP Manager: Failed to save headers (AJAX) - ' . $error_message . ', Project ID: ' . ($project_info['id'] ?? 'unknown') . ', Response Code: ' . $response_code);
            }
            wp_send_json_error($error_message);
        }
    }
    
    /**
     * Periodic header saving (called by WordPress cron)
     */
    public function save_header_history_periodic() {
        if (empty($this->options['api_key']) || empty($this->options['enabled'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP Manager: Periodic header save skipped - API key or plugin not enabled');
            }
            return;
        }
        
        // Get current headers
        $headers = $this->get_current_headers();
        
        // Get project info
        $project_info = $this->options['project_info'] ?? null;
        if (!$project_info || !isset($project_info['id'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP Manager: Periodic header save skipped - Project info not available');
            }
            return;
        }
        
        // Determine API base URL
        $is_local_dev = defined('WP_DEBUG') && WP_DEBUG && 
                        (strpos(home_url(), '127.0.0.1') !== false || 
                         strpos(home_url(), 'localhost') !== false ||
                         strpos(home_url(), '.local') !== false);
        
        if ($is_local_dev) {
            $api_base = 'http://host.docker.internal:8000/api';
        } else {
            $api_base = 'https://metricpoints.com/api';
        }
        
        $save_url = $api_base . '/v1/csp-projects/' . $project_info['id'] . '/header-history';
        
        $response = wp_remote_post($save_url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $this->options['api_key'],
                'User-Agent' => 'MetricPoints-WordPress-Plugin/' . METRICPOINTS_CSP_MANAGER_VERSION
            ),
            'body' => json_encode(array(
                'headers' => $headers,
                'csp_policy' => $this->get_current_csp_policy(),
                'site_url' => home_url(),
            ))
        ));
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints CSP Manager: Periodic header save failed - ' . $response->get_error_message() . ', Project ID: ' . $project_info['id']);
            }
        } else {
            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code !== 201) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    $response_body = wp_remote_retrieve_body($response);
                    error_log('MetricPoints CSP Manager: Periodic header save failed - Response Code: ' . $response_code . ', Project ID: ' . $project_info['id'] . ', Response: ' . $response_body);
                }
            } else {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('MetricPoints CSP Manager: Periodic header save successful - Project ID: ' . $project_info['id']);
                }
            }
        }
    }
    
    /**
     * Get current headers that would be sent
     */
    private function get_current_headers() {
        $headers = array();
        
        // CSP Policy
        $csp_policy = $this->get_current_csp_policy();
        if (!empty($csp_policy)) {
            $csp_mode = isset($this->options['csp_mode']) ? $this->options['csp_mode'] : 'enforcing';
            if ($csp_mode === 'report-only') {
                $headers['Content-Security-Policy-Report-Only'] = $csp_policy;
            } else {
                $headers['Content-Security-Policy'] = $csp_policy;
            }
        }
        
        // HSTS
        if (!empty($this->options['hsts_enabled'])) {
            $max_age = isset($this->options['hsts_max_age']) ? $this->options['hsts_max_age'] : 31536000;
            $hsts_value = "max-age={$max_age}";
            if (!empty($this->options['hsts_include_subdomains'])) {
                $hsts_value .= '; includeSubDomains';
            }
            if (!empty($this->options['hsts_preload'])) {
                $hsts_value .= '; preload';
            }
            $headers['Strict-Transport-Security'] = $hsts_value;
        }
        
        // X-Frame-Options
        if (!empty($this->options['x_frame_options'])) {
            $headers['X-Frame-Options'] = $this->options['x_frame_options'];
        }
        
        // X-Content-Type-Options
        if (!empty($this->options['x_content_type_options'])) {
            $headers['X-Content-Type-Options'] = 'nosniff';
        }
        
        // Referrer-Policy
        if (!empty($this->options['referrer_policy'])) {
            $headers['Referrer-Policy'] = $this->options['referrer_policy'];
        }
        
        // X-XSS-Protection
        if (!empty($this->options['x_xss_protection'])) {
            $headers['X-XSS-Protection'] = $this->options['x_xss_protection'];
        }
        
        // Permissions-Policy
        if (!empty($this->options['permissions_policy_enabled'])) {
            $features = isset($this->options['permissions_policy_features']) ? $this->options['permissions_policy_features'] : array();
            if (!empty($features)) {
                $policy_parts = array();
                foreach ($features as $feature => $value) {
                    if ($value === 'none') {
                        $policy_parts[] = $feature . '=()';
                    } elseif ($value === 'self') {
                        $policy_parts[] = $feature . '=(self)';
                    } elseif ($value === '*') {
                        $policy_parts[] = $feature . '=*';
                    }
                }
                if (!empty($policy_parts)) {
                    $headers['Permissions-Policy'] = implode(', ', $policy_parts);
                }
            }
        }
        
        // Expect-CT
        if (!empty($this->options['expect_ct_enabled'])) {
            $max_age = isset($this->options['expect_ct_max_age']) ? $this->options['expect_ct_max_age'] : 86400;
            $expect_ct_value = "max-age={$max_age}";
            if (!empty($this->options['expect_ct_enforce'])) {
                $expect_ct_value .= ', enforce';
            }
            if (!empty($this->options['expect_ct_report_uri'])) {
                $expect_ct_value .= ', report-uri="' . $this->options['expect_ct_report_uri'] . '"';
            }
            $headers['Expect-CT'] = $expect_ct_value;
        }
        
        // Clear-Site-Data
        if (!empty($this->options['clear_site_data_enabled'])) {
            $directives = isset($this->options['clear_site_data_directives']) ? $this->options['clear_site_data_directives'] : array();
            if (!empty($directives)) {
                $clear_directives = array();
                foreach ($directives as $directive => $enabled) {
                    if ($enabled) {
                        $clear_directives[] = '"' . $directive . '"';
                    }
                }
                if (!empty($clear_directives)) {
                    $headers['Clear-Site-Data'] = implode(', ', $clear_directives);
                }
            }
        }
        
        // Cross-Origin-Embedder-Policy (COEP)
        if (!empty($this->options['coep_enabled'])) {
            $headers['Cross-Origin-Embedder-Policy'] = isset($this->options['coep_value']) ? $this->options['coep_value'] : 'require-corp';
        }
        
        // Cross-Origin-Opener-Policy (COOP)
        if (!empty($this->options['coop_enabled'])) {
            $headers['Cross-Origin-Opener-Policy'] = isset($this->options['coop_value']) ? $this->options['coop_value'] : 'same-origin';
        }
        
        // Cross-Origin-Resource-Policy (CORP)
        if (!empty($this->options['corp_enabled'])) {
            $headers['Cross-Origin-Resource-Policy'] = isset($this->options['corp_value']) ? $this->options['corp_value'] : 'same-origin';
        }
        
        // Custom headers
        if (!empty($this->options['custom_headers'])) {
            $custom_headers = explode("\n", $this->options['custom_headers']);
            foreach ($custom_headers as $line) {
                $line = trim($line);
                if (empty($line) || strpos($line, '#') === 0) {
                    continue;
                }
                if (strpos($line, ':') !== false) {
                    $parts = explode(':', $line, 2);
                    $header_name = trim($parts[0]);
                    $header_value = trim($parts[1]);
                    if (!empty($header_name) && !empty($header_value)) {
                        $headers[$header_name] = $header_value;
                    }
                }
            }
        }
        
        return $headers;
    }
    
    /**
     * Verify API key with MetricPoints backend
     */
    private function verify_api_key($api_key) {
        if (empty($api_key)) {
            return array('success' => false, 'error' => 'API key is empty');
        }
        
        // Determine API base URL
        $is_local_dev = defined('WP_DEBUG') && WP_DEBUG && 
                        (strpos(home_url(), '127.0.0.1') !== false || 
                         strpos(home_url(), 'localhost') !== false ||
                         strpos(home_url(), '.local') !== false);
        
        if ($is_local_dev) {
            $api_base = 'http://host.docker.internal:8000/api';
        } else {
            $api_base = 'https://metricpoints.com/api';
        }
        
        $verify_url = $api_base . '/v1/verify-api-key/' . urlencode($api_key);
        
        $response = wp_remote_get($verify_url, array(
            'timeout' => 10,
            'headers' => array(
                'User-Agent' => 'MetricPoints-WordPress-Plugin/' . METRICPOINTS_CSP_MANAGER_VERSION
            )
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => 'Failed to connect to MetricPoints: ' . $response->get_error_message()
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        
        if ($response_code === 200 && isset($response_data['success']) && $response_data['success']) {
            return array(
                'success' => true,
                'project' => $response_data['project'] ?? null,
                'team' => $response_data['team'] ?? null,
            );
        } else {
            return array(
                'success' => false,
                'error' => $response_data['error'] ?? 'Invalid API key'
            );
        }
    }
    
    /**
     * Sync policy update to MetricPoints
     */
    private function sync_policy_update_to_metricpoints($old_policy, $new_policy, $update_type = 'manual') {
        $api_key = $this->options['api_key'] ?? '';
        $project_info = $this->options['project_info'] ?? null;
        
        if (empty($api_key) || empty($project_info) || !isset($project_info['id'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints: Cannot sync policy update - missing API key or project info');
            }
            return false;
        }
        
        // Determine API base URL
        $is_local_dev = defined('WP_DEBUG') && WP_DEBUG && 
                        (strpos(home_url(), '127.0.0.1') !== false || 
                         strpos(home_url(), 'localhost') !== false ||
                         strpos(home_url(), '.local') !== false);
        
        if ($is_local_dev) {
            $api_base = 'http://host.docker.internal:8000/api';
        } else {
            $api_base = 'https://metricpoints.com/api';
        }
        
        $sync_url = $api_base . '/v1/csp-projects/' . urlencode($project_info['id']) . '/policy-updates';
        
        $response = wp_remote_post($sync_url, array(
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $api_key,
                'User-Agent' => 'MetricPoints-WordPress-Plugin/' . METRICPOINTS_CSP_MANAGER_VERSION
            ),
            'body' => json_encode(array(
                'previous_policy' => $old_policy,
                'new_policy' => $new_policy,
                'update_type' => $update_type,
                'client_info' => array(
                    'client_type' => 'wordpress',
                    'client_version' => METRICPOINTS_CSP_MANAGER_VERSION,
                    'platform' => 'WordPress ' . get_bloginfo('version'),
                    'site_url' => home_url(),
                )
            ))
        ));
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints: Failed to sync policy update - ' . $response->get_error_message());
            }
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code >= 200 && $response_code < 300) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints: Policy update synced successfully');
            }
            return true;
        } else {
            $response_body = wp_remote_retrieve_body($response);
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('MetricPoints: Failed to sync policy update - HTTP ' . $response_code . ': ' . $response_body);
            }
            return false;
        }
    }
    
    /**
     * Get project ID from API key verification
     */
    private function get_project_id() {
        $project_info = $this->options['project_info'] ?? null;
        return $project_info['id'] ?? null;
    }
    
    /**
     * AJAX handler for verifying connection
     */
    public function handle_verify_connection() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'metricpoints_csp_manager_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        $api_key = $this->options['api_key'] ?? '';
        
        if (empty($api_key)) {
            wp_send_json_error('API key not configured');
        }
        
        $verification = $this->verify_api_key($api_key);
        
        if ($verification['success']) {
            // Update stored project info
            $this->options['project_info'] = $verification['project'] ?? null;
            $this->options['connection_status'] = 'connected';
            $this->options['connection_verified_at'] = current_time('mysql');
            update_option('metricpoints_csp_manager_options', $this->options);
            
            wp_send_json_success(array(
                'message' => '✅ Connection verified successfully!',
                'project' => $verification['project'] ?? null,
            ));
        } else {
            wp_send_json_error(array(
                'message' => '❌ Connection failed: ' . ($verification['error'] ?? 'Unknown error'),
            ));
        }
    }
}

/**
 * Plugin activation hook - runs when plugin is activated
 */
register_activation_hook(__FILE__, 'metricpoints_csp_manager_activate');

function metricpoints_csp_manager_activate() {
    // Set default options if they don't exist
    $default_options = array(
        'enabled' => 0, // Start disabled so user can configure first
        'api_key' => '',
        'self_healing' => 0, // Start disabled
        'csp_policy' => "script-src 'self'; object-src 'none';",
        'learning_mode' => 1,
        'auto_update_threshold' => 0.8,
        'max_violations_per_hour' => 100
    );
    
    $existing_options = get_option('metricpoints_csp_manager_options', array());
    $merged_options = array_merge($default_options, $existing_options);
    update_option('metricpoints_csp_manager_options', $merged_options);
    
    // Initialize empty violations array
    if (!get_option('metricpoints_csp_violations')) {
        update_option('metricpoints_csp_violations', array());
    }
    
    // Initialize empty policy backups array
    if (!get_option('metricpoints_csp_policy_backups')) {
        update_option('metricpoints_csp_policy_backups', array());
    }
    
    // Initialize update count
    if (!get_option('metricpoints_csp_update_count')) {
        update_option('metricpoints_csp_update_count', 0);
    }
    
    // Flush rewrite rules to ensure REST API endpoints are registered
    flush_rewrite_rules();
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('MetricPoints CSP Manager: Plugin activated successfully');
    }
}

/**
 * Plugin deactivation hook - runs when plugin is deactivated
 */
register_deactivation_hook(__FILE__, 'metricpoints_csp_manager_deactivate');

function metricpoints_csp_manager_deactivate() {
    // Flush rewrite rules
    flush_rewrite_rules();
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('MetricPoints CSP Manager: Plugin deactivated');
    }
}

/**
 * Plugin uninstall hook - runs when plugin is deleted
 */
register_uninstall_hook(__FILE__, 'metricpoints_csp_manager_uninstall');

function metricpoints_csp_manager_uninstall() {
    // Optionally remove all plugin data
    // delete_option('metricpoints_csp_manager_options');
    // delete_option('metricpoints_csp_violations');
    // delete_option('metricpoints_csp_policy_backups');
    // delete_option('metricpoints_csp_update_count');
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('MetricPoints CSP Manager: Plugin uninstalled');
    }
}

// Initialize the plugin
new MetricPointsCSPManager();
