/**
 * MetricPoints CSP Manager for WordPress
 * Handles CSP violation reporting and self-healing
 */

(function() {
    'use strict';
    
    // Configuration from WordPress
    if (typeof metricpointsCspManager === 'undefined') {
        return;
    }
    
    const config = metricpointsCspManager;
    
    // Initialize CSP violation reporting
    function init() {
        // Set up CSP violation reporting
        setupCSPViolationReporting();
    }
    
    // Set up CSP violation reporting
    function setupCSPViolationReporting() {
        // Create a script to handle CSP violations
        const script = document.createElement('script');
        script.textContent = `
            (function() {
                // Override console.error to catch CSP violations
                const originalError = console.error;
                console.error = function() {
                    // Check if this is a CSP violation
                    const message = arguments[0];
                    if (typeof message === 'string' && message.includes('Content Security Policy')) {
                        reportCSPViolation(message);
                    }
                    originalError.apply(console, arguments);
                };
                
                // Listen for CSP violation events
                document.addEventListener('securitypolicyviolation', function(event) {
                    reportCSPViolationEvent(event);
                });
                
                function reportCSPViolation(message) {
                    // Parse CSP violation message
                    const violation = parseCSPViolationMessage(message);
                    if (violation) {
                        sendViolationReport(violation);
                    }
                }
                
                function reportCSPViolationEvent(event) {
                    const violation = {
                        'violated-directive': event.violatedDirective,
                        'blocked-uri': event.blockedURI,
                        'document-uri': event.documentURI,
                        'source-file': event.sourceFile,
                        'line-number': event.lineNumber,
                        'column-number': event.columnNumber,
                        'status-code': event.statusCode,
                        'user-agent': navigator.userAgent,
                        'timestamp': new Date().toISOString()
                    };
                    
                    sendViolationReport(violation);
                }
                
                function parseCSPViolationMessage(message) {
                    // Simple parsing of CSP violation messages
                    const directiveMatch = message.match(/Refused to ([^']+) '([^']+)'/);
                    if (directiveMatch) {
                        return {
                            'violated-directive': directiveMatch[1],
                            'blocked-uri': directiveMatch[2],
                            'document-uri': window.location.href,
                            'source-file': window.location.href,
                            'line-number': 0,
                            'column-number': 0,
                            'status-code': 0,
                            'user-agent': navigator.userAgent,
                            'timestamp': new Date().toISOString()
                        };
                    }
                    return null;
                }
                
                function sendViolationReport(violation) {
                    // Send to WordPress backend
                    fetch('${config.ajaxUrl}', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'action=metricpoints_csp_violation&nonce=${config.nonce}&violation=' + encodeURIComponent(JSON.stringify(violation))
                    }).catch(error => {
                        console.warn('Failed to send CSP violation report:', error);
                    });
                }
            })();
        `;
        
        document.head.appendChild(script);
    }
    
    // Test CSP violation function (only available in debug mode)
    window.testCSPViolation = function() {
        // Try to load an external script (should be blocked by CSP)
        const script = document.createElement('script');
        script.src = 'https://example.com/test-script.js';
        script.onerror = function() {
            // Violation will be reported via securitypolicyviolation event
        };
        document.head.appendChild(script);
        
        // Try to execute inline script (should be blocked if CSP doesn't allow it)
        try {
            eval('void(0);');
        } catch (e) {
            // Violation will be reported via securitypolicyviolation event
        }
    };
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
})();
